###############################################
##      Filename:     builder_scene.py      ##
##      Date:         18/04/2020             ##
##      Author:       Ayman HATOUM           ##
###############################################
"""This file contains the class definition of builderScene(). A subclass
of QGraphicsScene() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtCore import Qt
from PyQt5.QtCore import QPoint
from PyQt5.QtCore import QPointF
from PyQt5.QtCore import pyqtSignal
from PyQt5.QtWidgets import QGraphicsScene
from PyQt5.QtWidgets import QGraphicsItem
from PyQt5.QtWidgets import QMenu
from PyQt5.QtWidgets import QGraphicsView
from PyQt5.QtGui import QBrush
from PyQt5.QtGui import QTransform
from PyQt5.QtGui import QGuiApplication
from src.tools import taskSetItem
from src.dialogs.task_parameters_dialog import taskParametersDialog
from src.windows.central.builder.scene.items.task_item import taskItem
from src.windows.central.builder.scene.items.task_port_item import taskPortItem
from src.windows.central.builder.scene.items.connection_item import connectionItem

#----------------------------------------------------------------------------#

class builderScene(QGraphicsScene) :
    "Builder class scene"
    DefaultMode, ConnectingMode = range(2)
    rebuildDesign = pyqtSignal()
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing builder's scene...")
        super().__init__(parent)
        
        self.view = parent
        
        self.tasks = []
        self.connections = []
        self.mode = self.DefaultMode
        
        self.setBuilderSceneOptions()  
        
        self.view.mainWindow.targets.tree.targetRemoved.connect(self.targetRemovedFromTree)
        
    def setBuilderSceneOptions(self) :
        self.backgroundPattern = QBrush(Qt.lightGray, Qt.Dense7Pattern)
        self.setBackgroundBrush(self.backgroundPattern)

    def clearScene(self) :
        logging.getLogger(__name__).debug("Clearing builder's scene...")
        self.clear()
        self.tasks = []
        self.connections = []
        self.mode = self.DefaultMode
        taskItem.count = 0
        self.view.setWindowModified(False)
        self.view.loadingFile = True

    def loadScene(self, scene) :
        logging.getLogger(__name__).debug("Loading builder's scene...")
        for item in scene["tasks"] :
            task = self.view.mainWindow.targets.tree.getTask(item["target index"], item["name"])
            pos = QPointF(item["scene position"][0], item["scene position"][1])
            try :
                defaults = item["default inputs"]
            except KeyError :
                defaults = None
            try :
                label = item["label"]
            except KeyError :
                label = None
            self.loadTask(task, pos, item["id"], label, defaults)
        for item in scene["connections"] :
            outTask = self.taskFromId(item["outtask"])
            outPort = outTask.outputFromLabel(item["output"])
            inTask = self.taskFromId(item["intask"])
            inPort = inTask.inputFromLabel(item["input"])
            self.loadConnection(outPort, inPort)
        self.view.setWindowModified(False)
        self.view.loadingFile = True

    def taskFromId(self, id) :
        logging.getLogger(__name__).debug("Getting task from ID: [{}]...".format(id))
        for task in self.tasks :
            if task.id == id :
                return task
        return None

    def loadConnection(self, outPort, inPort) :
        logging.getLogger(__name__).debug("Loading connection...")
        connection = connectionItem(outPort, True, True, inPort)
        self.connections.append(connection)
        self.addItem(connection)

    def saveScene(self) :
        logging.getLogger(__name__).debug("Saving builder's scene...")
        self.view.setWindowModified(False)
        data = {}
        tasksData = {}
        tasksData["tasks"] = []
        connectionsData = {}
        connectionsData["connections"] = []
        data["scene"] = {**tasksData, **connectionsData}
        if self.tasks :
            for item in self.tasks :
                itemData = {}
                itemData["name"] = item.name
                itemData["target index"] = item.task.treeWidget().indexOfTopLevelItem(item.task.parent())
                itemData["id"] = item.id
                if item.label :
                    itemData["label"] = item.label.toPlainText()
                itemData["scene position"] = [item.scenePos().x(), item.scenePos().y()]
                if item.inputs :
                    itemData["default inputs"] = {}  
                    for port in item.inputs :
                        itemData["default inputs"][port.label] = port.defaultValue    
                tasksData["tasks"].append(itemData)
        if self.connections :
            for item in self.connections :
                itemData = {}
                itemData["outtask"] = item.outPort.task.id
                itemData["output"] = item.outPort.label  
                itemData["intask"] = item.inPort.task.id
                itemData["input"] = item.inPort.label
                connectionsData["connections"].append(itemData)  
        return data

    def builderSanityCheck(self) :
        logging.getLogger(__name__).debug("Checking builder's sanity...")
        if self.tasks :
            errorMessage = "Builder design doesn't meet sanity requirements!\n\n{}\n"
            for item in self.tasks :
                for param in item.inputs :
                    if param.connection == None :
                        logging.getLogger(__name__).warning(errorMessage.format("Input ports should all to be connected."))
                        return False

                for param in item.outputs :
                    if param.connections == [] :
                        logging.getLogger(__name__).warning(errorMessage.format("Output ports should all to be connected."))
                        return False

            for item in self.connections :
                if not item.valid :
                    logging.getLogger(__name__).warning(errorMessage.format("Remove broken connections due to type mismatch."))
                    return False

        return True    

    def extractBuilderData(self) :
        logging.getLogger(__name__).debug("Extracting builder's data...")
        data = []
        for item in self.tasks :
            targetID = item.task.parent().canID
            taskID = item.id
            taskWCET = item.task.taskData.wcet
            inputs = []
            for param in item.inputs :
                #supposing everything is connected.. i.e a sanity is run before calling this function
                inputType = param.typ
                connectedToTaskID = param.connection.outPort.task.id
                connectedToTaskPort = param.connection.outPort.label
                inputs.append((inputType, connectedToTaskID, connectedToTaskPort))
            outputs = []
            for param in item.outputs :
                outputType = param.typ
                #supposing everything is connected.. i.e a sanity is run before calling this function
                connectedTo = []
                for conn in param.connections :
                    connectedToTaskID = conn.inPort.task.id
                    connectedTo.append(connectedToTaskID)
                outputs.append((outputType, connectedTo))
            data.append(taskSetItem(targetID, taskID, taskWCET, inputs, outputs))

        return data

    def targetRemovedFromTree(self, target) :
        logging.getLogger(__name__).debug("Removing children tasks of removed target {}...".format(target))
        targetTasks = []
        for item in self.tasks :
            if item.task.parent() == target :
                targetTasks.append(item)
        for item in targetTasks :
            self.removeTask(item)

    def loadTask(self, task, pos, id, label, defaults) :
        logging.getLogger(__name__).debug("Loading task graphics item...")
        item = taskItem(task, True, id, label, defaults)
        self.addItem(item)
        item.setPos(pos - QPoint(50, 35))
        self.tasks.append(item)
    
    def addTask(self, task, pos) :
        logging.getLogger(__name__).debug("Adding task {} on scene...".format(task))
        item = taskItem(task)
        self.addItem(item)
        item.setPos(pos - QPoint(50, 35))
        self.tasks.append(item)
        self.rebuildDesign.emit()
        
    def dragEnterEvent(self, event) :
        if event.mimeData().hasFormat("application/x-qtreewidgetitemtask") :
            event.accept()
        else :
            event.ignore()
            
    def dragMoveEvent(self, event) :
        if event.mimeData().hasFormat("application/x-qtreewidgetitemtask") :
            event.accept()
        else :
            event.ignore()
            
    def dropEvent(self, event) :
        logging.getLogger(__name__).debug("Dropping event invoked...")
        mimeData = event.mimeData()
        task = mimeData.taskItem
        self.addTask(task, event.scenePos())
        event.acceptProposedAction()
        
    def contextMenuEvent(self, event) :
        logging.getLogger(__name__).debug("Context menu event invoked...")
        contextMenu = QMenu()        
        taskParametersAction = contextMenu.addAction("Modify Parameters")     
        taskParametersAction.triggered.connect(self.showTaskParametersDialog)     
        taskParametersAction.setEnabled(False)
        contextMenu.addSeparator()
        contextMenu.addAction(self.view.mainWindow.allActions.removeSelectedAction)
                
        selectedItems = self.selectedItems()
        if len(selectedItems) == 1 and selectedItems[0].type() == QGraphicsItem.UserType : 
            taskParametersAction.setEnabled(True)
                
        contextMenu.exec(event.screenPos())

    def showTaskParametersDialog(self) :
        selectedItem = self.selectedItems()[0]
        logging.getLogger(__name__).debug("Task parameters action triggered on {}...".format(selectedItem))
        dialog = taskParametersDialog(self.view.mainWindow, selectedItem)
        if dialog.exec() :
            label = dialog.data.pop("Task label")
            selectedItem.updateLabel(label)
            if dialog.data :
                selectedItem.setInputsDefaultValues(dialog.data)
    
    def removeTask(self, task) :
        logging.getLogger(__name__).debug("Removing task {}...".format(task))
        if task.inputs :
            for port in task.inputs :
                if port.connection :
                    self.removeConnection(port.connection)
        if task.outputs :
            for port in task.outputs :
                if port.connections :
                    for connection in port.connections[:] : 
                        self.removeConnection(connection)                    
        self.removeItem(task)
        self.tasks.remove(task)
        self.rebuildDesign.emit()
        
    def removeConnection(self, connection) :
        logging.getLogger(__name__).debug("Removing connection {}...".format(connection))
        connection.outPort.connections.remove(connection)
        connection.inPort.connection = None
        self.removeItem(connection)
        self.connections.remove(connection)
        self.rebuildDesign.emit()
        
    def setMode(self, mode) :
        logging.getLogger(__name__).debug("Setting mode: [{}]...".format(mode))
        self.mode = mode
        
    def mouseMoveEvent(self, event) :
        if self.mode == self.ConnectingMode :
            self.connection.setTail(event.scenePos())
            self.connection.update()
        else :
            QGraphicsScene.mouseMoveEvent(self, event)
        
    def mousePressEvent(self, event) :
        if self.mode == self.ConnectingMode :
            if event.button() == Qt.LeftButton :
                item = self.itemAt(event.scenePos(), QTransform())
                if item :   
                    if item.type() == QGraphicsItem.Type + 3 and item.isEligible(self.connection.sourcePort()) \
                            and not self.connectionTask.isAncestorOf(item) :
                        if item.direction == taskPortItem.Input :
                            if item.connection == None :
                                self.connection.latch(item)                        
                                item.connection = self.connection
                                self.connections.append(self.connection)
                                self.exitConnectionMode()
                        else :
                            self.connection.latch(item)
                            item.connections.append(self.connection)
                            self.connections.append(self.connection)
                            self.exitConnectionMode()
                        self.rebuildDesign.emit()                         
        else :
            if event.button() == Qt.LeftButton :
                item = self.itemAt(event.scenePos(), QTransform())
                if item :
                    if item.type() == QGraphicsItem.Type + 3 :
                        if item.direction == taskPortItem.Input :
                            if item.connection == None :
                                self.enterConnectionMode()
                                self.connection = connectionItem(item, False)
                                self.connectionTask = item.task
                                self.addItem(self.connection)
                        else :
                            self.enterConnectionMode()
                            self.connection = connectionItem(item, True)
                            self.connectionTask = item.task
                            self.addItem(self.connection)                            
    
            QGraphicsScene.mousePressEvent(self, event)
        
    def mouseReleaseEvent(self, event) :
        QGraphicsScene.mouseReleaseEvent(self, event)
        
    def keyPressEvent(self, event) :
        if self.mode == self.ConnectingMode :
            if event.key() == Qt.Key_Escape :
                self.removeItem(self.connection)
                self.exitConnectionMode()
        else :
            if event.key() == Qt.Key_Down :
                selectedTasks = []
                for item in self.selectedItems() :
                    if item.type() == QGraphicsItem.UserType :
                        selectedTasks.append(item) 
                if selectedTasks :
                    logging.getLogger(__name__).debug("Moving selected: {} down...".format(selectedTasks))
                    group = self.createItemGroup(selectedTasks)
                    group.moveBy(0, 10)
                    self.destroyItemGroup(group)
                    return
            if event.key() == Qt.Key_Up :
                selectedTasks = []
                for item in self.selectedItems() :
                    if item.type() == QGraphicsItem.UserType :
                        selectedTasks.append(item)                
                if selectedTasks :
                    logging.getLogger(__name__).debug("Moving selected: {} up...".format(selectedTasks))
                    group = self.createItemGroup(selectedTasks)
                    group.moveBy(0, -10)
                    self.destroyItemGroup(group)
                    return
            if event.key() == Qt.Key_Right :
                selectedTasks = []
                for item in self.selectedItems() :
                    if item.type() == QGraphicsItem.UserType :
                        selectedTasks.append(item)                
                if selectedTasks :
                    logging.getLogger(__name__).debug("Moving selected: {} right...".format(selectedTasks))
                    group = self.createItemGroup(selectedTasks)
                    group.moveBy(10, 0)
                    self.destroyItemGroup(group)
                    return
            if event.key() == Qt.Key_Left :
                selectedTasks = []
                for item in self.selectedItems() :
                    if item.type() == QGraphicsItem.UserType :
                        selectedTasks.append(item)                
                if selectedTasks :
                    logging.getLogger(__name__).debug("Moving selected: {} left...".format(selectedTasks))
                    group = self.createItemGroup(selectedTasks)
                    group.moveBy(-10, 0)
                    self.destroyItemGroup(group)
                    return

        QGraphicsScene.keyPressEvent(self, event)

    def keyReleaseEvent(self, event) :
        QGraphicsScene.keyReleaseEvent(self, event)

    def exitConnectionMode(self) :
        logging.getLogger(__name__).debug("Exiting connecting mode, connection: {} task: {}...".format(self.connection, self.connectionTask))
        del self.connection
        del self.connectionTask
        self.mode = self.DefaultMode
        QGuiApplication.restoreOverrideCursor()
        self.view.mainWindow.allActions.modeToolActionGroup.setEnabled(True)
        self.view.setDragMode(self.restoreDragMode)
        
    def enterConnectionMode(self) :
        logging.getLogger(__name__).debug("Entering connecting mode...")
        self.mode = self.ConnectingMode
        self.restoreDragMode = self.view.dragMode()
        self.view.setDragMode(QGraphicsView.NoDrag)
        QGuiApplication.setOverrideCursor(Qt.CrossCursor)
        self.view.mainWindow.allActions.modeToolActionGroup.setEnabled(False)      
                
    def focusOutEvent(self, event) :
        if self.mode == self.ConnectingMode :
            self.removeItem(self.connection)
            self.exitConnectionMode()
        QGraphicsScene.focusOutEvent(self, event)

#----------------------------------------------------------------------------#