################################################
##      Filename:     builder_window.py      ##
##      Date:         18/04/2020              ##
##      Author:       Ayman HATOUM            ##
################################################
"""This file contains the class definition of builderWindow(). A subclass
of QGraphicsView() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
import math
from PyQt5.QtCore import Qt
from PyQt5.QtWidgets import QGraphicsView
from src.constants import whatsThisMsg
from src.windows.central.builder.scene.builder_scene import builderScene

#----------------------------------------------------------------------------#

class builderWindow(QGraphicsView) :
    "Builder class window"
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing builder's view...")
        super().__init__(parent)
        
        self.mainWindow = parent
        self.zoom = 100
        self.createLayout()

        self.scene.changed.connect(self.sceneModified)
        self.loadingFile = False

        self.setWhatsThis(whatsThisMsg.format("Builder Window", "Here you can build all your project, establishing all desired data dependencies by connecting tasks together. You can navigate using your <i>Cursor Keys</i>. You can zoom using your <i>Mouse Wheel</i>. You can <i>Move</i> | <i>Select</i> | <i>Remove</i> any task. You can <i>Modify Parameters</i> for any task <strong>Double Click</strong> | <strong>Builder Context Menu -> Modify Parameters</strong>."))
        
    def sceneModified(self, rect) :
        if self.isWindowModified() :
            return
        if self.loadingFile :
            self.loadingFile = False
            return
        if len(rect) == 1 :
            if rect[0].isNull() :
                return
        self.setWindowModified(True)
        self.mainWindow.setWindowModified(True)

    def createLayout(self) :
        self.scene = builderScene(self)
        self.setScene(self.scene)
        self.setDragMode(QGraphicsView.RubberBandDrag)
        self.setTransformationAnchor(QGraphicsView.AnchorUnderMouse)
        
    def mouseMoveEvent(self, event) :
        QGraphicsView.mouseMoveEvent(self, event)
        
    def mousePressEvent(self, event) :
        QGraphicsView.mousePressEvent(self, event)
        
    def mouseReleaseEvent(self, event) :
        QGraphicsView.mouseReleaseEvent(self, event)

    def zoomIn(self) :
        logging.getLogger(__name__).debug("Zooming in...")
        if self.zoom > 300 :
            return 
        zoomInFactor = 1.1
        self.scale(zoomInFactor, zoomInFactor)
        self.zoom = self.zoom*zoomInFactor
        self.mainWindow.toolBar_.updateZoomButt(int(self.zoom))

    def zoomOut(self) :
        logging.getLogger(__name__).debug("Zooming out...")
        if self.zoom < 30 :
            return
        zoomOutFactor = 1 / 1.1
        self.scale(zoomOutFactor, zoomOutFactor)
        self.zoom = self.zoom*zoomOutFactor
        self.mainWindow.toolBar_.updateZoomButt(int(self.zoom))

    def resetZoom(self) :
        logging.getLogger(__name__).debug("Resetting zoom...")
        if self.zoom > 100 :
            zoomFactor = 1 / 1.1
        elif self.zoom < 100 :
            zoomFactor = 1.1
        else :
            return
        recursionFactor = math.log(100/self.zoom)/math.log(zoomFactor)
        self.scale(math.pow(zoomFactor, recursionFactor), math.pow(zoomFactor, recursionFactor))
        self.zoom = 100
        self.mainWindow.toolBar_.updateZoomButt(int(self.zoom))

    def wheelEvent(self, event) :
        logging.getLogger(__name__).debug("Wheel event invoked...")
        if event.modifiers() & Qt.ControlModifier :
            delta = event.angleDelta().y()
            if delta > 0 :
                self.zoomIn()
            elif delta < 0 :
                self.zoomOut()
            return                
        QGraphicsView.wheelEvent(self, event)           

#----------------------------------------------------------------------------#