##########################################
##      Filename:     tools.py          ##
##      Date:         12/05/2020        ##
##      Author:       Ayman HATOUM      ##
##########################################
"""This file contains the tool functions used in the whole project"""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
# from datetime import datetime
import collections
import itertools
import pulp
import json
import logging
import pathlib
import jsonschema
from PyQt5.QtCore import QThread
from PyQt5.QtCore import pyqtSignal
from src.constants import floatBytes
from src.constants import intBytes
from src.constants import boolBytes
from src.constants import schemaPath
from src.constants import targetsPath
from src.constants import workingDirectory
from src.windows.targets.target_class import targetClass

#----------------------------------------------------------------------------#

#task type
FreeTask, ConsumerTask, ProducerTask = range(3)
#used to extract raw data from the builder
taskSetItem = collections.namedtuple("Task", "targetID, taskID, WCET, inputs, outputs")
#used for demonstrating the data after running a check on all possible dependencies
taskItemTransformation = collections.namedtuple("TT", "taskID, WCET, Tx, slotBuff, interDependencies, consumerDependencies, producerDependencies")
#used for returning scheduled tasks
taskScheduled = collections.namedtuple("ST", "taskID, offset, deadline, WCET, Tx, slotBuff, taskType")
#used for returning timeline scheduled tasks
taskTimeline = collections.namedtuple("Timeline", "name, color, offset, deadline, WCET, Tx, taskType")
#used for extracting slots
timeSlot = collections.namedtuple("Slot", "start, overlapBound, buffStart, end, tasks")
#used for demonstrating a task in a slot
timeSlotTask = collections.namedtuple("SlotTask", "targetID, taskID, offset, deadline, WCET, Tx, taskType")

def modeling(data, period) :
    "Models the passed data into an LP optimization problem"
    taskIDs = []                                #to define the indexes of our decision variables
    taskTUFs = []                               #to define our objective function
    targetTaskIDs = {}                          #to define the constraints on each target, i.e not overlapping
    producersTaskIDTxs = {}                     #to define the constraints between producers, i.e not overlapping slots
    targetTaskIDCombinations = []               #to define binary variable for modeling the logic of OR into an inequality constraint between interdependencies
    producersTaskIDCombinations = []            #to define binary variable for modeling the logic of OR into an inequality constraint between producers of distinct targets
    producersWithRestTaskIDCombinations = []    #to define binary variable for modeling the logic of OR into an inequality constraint between producers and rest of tasks

    #define all TUFs, extract all taskIDs and their combinations on each target
    for target, taskSet in data.items() :
        targetTaskIDs[target] = list(map(lambda x: x.taskID, taskSet))
        targetTaskIDs[target].sort()
        producersTaskIDTxs.update(dict(map(lambda x: (x.taskID, x.Tx + x.slotBuff), filter(lambda x: x.producerDependencies, taskSet))))
        taskSetTUFs = [TimeUtilityFunction(task.taskID, task.WCET, period, task.producerDependencies != []) for task in taskSet]
        taskIDs.extend(targetTaskIDs[target])
        taskTUFs.extend(taskSetTUFs)
        targetTaskIDCombinations.extend(list(itertools.combinations(targetTaskIDs[target], 2)))
    
    #extract all combinations between producers
    producersTaskIDs = list(producersTaskIDTxs.keys())
    producersTaskIDs.sort()
    producersTaskIDCombinations.extend(list(itertools.combinations(producersTaskIDs, 2)))
    #extract all combinations between producers and non-producers
    producersWithRestTaskIDCombinations.extend(list(itertools.product(producersTaskIDs, list(set(taskIDs) - set(producersTaskIDs)))))
    
    #define optimization problem
    problem = pulp.LpProblem("optimizeOffsetsAndDeadlines", pulp.LpMaximize)
    #bound for variables
    offsetLowerBounds = 0
    deadlineUpperBounds = period
    #define variables: offsets, deadlines & binaries
    offsetVars = pulp.LpVariable.dicts("offset", taskIDs, lowBound = offsetLowerBounds, cat = pulp.LpInteger)
    deadlineVars = pulp.LpVariable.dicts("deadline", taskIDs, upBound = deadlineUpperBounds, cat = pulp.LpInteger)
    interBinaryVars = pulp.LpVariable.dicts("interBinary", targetTaskIDCombinations, cat = pulp.LpBinary)
    producerBinaryVars = pulp.LpVariable.dicts("producerBinary", producersTaskIDCombinations, cat = pulp.LpBinary)
    producerWithRestBinaryVars = pulp.LpVariable.dicts("producerWithRestBinary", producersWithRestTaskIDCombinations, cat = pulp.LpBinary)
    #define objective function as summation of all TUFs
    problem += pulp.lpSum([TUF.offset(offsetVars[TUF.ID]) for TUF in taskTUFs] + [TUF.deadline(deadlineVars[TUF.ID]) for TUF in taskTUFs])
    
    for target, taskSet in data.items() :
        for task in taskSet :
            #define bounds for variables
            offsetVars[task.taskID].upBound = period - task.WCET
            deadlineVars[task.taskID].lowBound = task.WCET
            
            #define overlapping on same target constraints
            for j in [x for x in targetTaskIDs[target] if x > task.taskID] :
                #task.taskID plus its WCET happens before task_j, OR..
                problem += offsetVars[task.taskID] + task.WCET - offsetVars[j] - period*interBinaryVars[(task.taskID, j)] <= 0
            for i in [x for x in targetTaskIDs[target] if x < task.taskID] :
                #task_i happens after task.taskID plus its WCET
                problem += offsetVars[task.taskID] + task.WCET - offsetVars[i] - period*(1 - interBinaryVars[(i, task.taskID)]) <= 0
            
            #define constraint for deadline to be equal to its offset plus its WCET
            problem += offsetVars[task.taskID] - deadlineVars[task.taskID] + task.WCET == 0

            #define constraints of producer slots on targets
            if task.taskID in producersTaskIDs :
                for j in [x for x in producersTaskIDs if x > task.taskID] :
                    #task.taskID plus its WCET plus its Tx happens before task_j, OR..
                    problem += offsetVars[task.taskID] + task.WCET + task.Tx + task.slotBuff - offsetVars[j] - period*producerBinaryVars[(task.taskID, j)] <= 0
                for i in [x for x in producersTaskIDs if x < task.taskID] :
                    #task_i happens after task.taskID plus its WCET plus its Tx
                    problem += offsetVars[task.taskID] + task.WCET + task.Tx + task.slotBuff - offsetVars[i] - period*(1 - producerBinaryVars[(i, task.taskID)]) <= 0
            # #define constraints of producer with rest of task in order not to surpass a slot end
            else :
                for j in producersTaskIDs :
                    #task.taskID plus its WCET happens before slot end of producer j, OR..
                    problem += offsetVars[task.taskID] + task.WCET - producersTaskIDTxs[j] - deadlineVars[j] - period*producerWithRestBinaryVars[(j, task.taskID)] <= 0
                    #task.taskID plus its WCET happens after slot end of producer j
                    problem += producersTaskIDTxs[j] + deadlineVars[j] - offsetVars[task.taskID] - period*(1 - producerWithRestBinaryVars[(j, task.taskID)]) <= 0

            #define constraints comming due interdependencies
            for param in task.interDependencies :
                #offset_task.taskID >= deadline_param
                problem += deadlineVars[param[1]] - offsetVars[task.taskID] <= 0
            #define constraints comming due producer dependencies
            for param in task.producerDependencies :
                #deadline_task.taskID + dT_task.taskID <= offset_param
                problem += deadlineVars[task.taskID] - offsetVars[param] + task.Tx + task.slotBuff <= 0

    return problem

def optimize(problem, data) :
    "Solves the passed optimization problem and returns results"
    #solve problem with default optimizer
    logging.getLogger(__name__).debug("Solver started...")
    problem.solve()
    logging.getLogger(__name__).debug("Solver ended...")

    # problem.solve(pulp.COIN(path = pulp.pulp_cbc_path, maxSeconds = 120))

    # # problem.writeLP("opti.lp")
    # t0 = datetime.now().time()
    # print(t0)
    # problem.solve(pulp.CHOCO_CMD(path = "C:\\tools\\Python38\\Lib\\site-packages\\pulp\\solverdir\\choco\\choco-parsers-with-dependencies.jar"))
    # # print(pulp.COIN().defaultPath())
    # print("pulp_cbc_path: ", pulp.pulp_cbc_path)
    # # print("pulp_choco_path: ", pulp.pulp_choco_path)
    # print("cbc_path: ", pulp.cbc_path)
    # # print("gurobi_path: ", pulp.gurobi_path)
    # # print("coinMP_path: ", pulp.coinMP_path)
    # t1 = datetime.now().time()
    # print(t1)

    period = 0

    status = pulp.LpStatus[problem.status]
    if status != "Optimal" :
        return (False, {}, period)
    
    scheduledTasks = {ID:[] for ID in data.keys()}
    
    decisionVariables = problem.variablesDict()
    for target, taskSet in data.items() :
        for task in taskSet :
            taskID = task.taskID
            offset = decisionVariables["offset_{}".format(taskID)].varValue
            deadline = decisionVariables["deadline_{}".format(taskID)].varValue
            if deadline > period :
                period = deadline
            WCET = task.WCET
            Tx = task.Tx
            slotBuff = task.slotBuff
            taskType = FreeTask | ConsumerTask if task.consumerDependencies != [] else FreeTask
            taskType = taskType | ProducerTask if task.producerDependencies != [] else taskType
            scheduledTasks[target].append(taskScheduled(taskID, offset, deadline, WCET, Tx, slotBuff, taskType))
        scheduledTasks[target] = sorted(scheduledTasks[target], key = lambda x: x.offset)

    return (True, scheduledTasks, period)

    #display problem and solution
    # print("\n\n")
    # if pulp.LpStatus[problem.status] == 'Infeasible' :
    #     print("The problem has no FEASIBLE solution under such constraints!")
    #     return False
    # elif pulp.LpStatus[problem.status] == 'Unbounded' :
    #     print("The objective function of the problem is UNBOUNDED!")
    #     return False
    # elif pulp.LpStatus[problem.status] == 'Undefined' :
    #     print("Could NOT find a feasible solution, BUT it may exist!")
    #     return False
    # elif pulp.LpStatus[problem.status] == 'Not Solved' :
    #     print("RUN problem.solve() to get a solution!")
    #     return False
    # else :
    #     print("Objective function: {}\n".format(problem.objective))
    #     print("Subject to:")
    #     for cons, eqt in problem.constraints.items() :
    #         print("constraint{} : {}".format(cons, eqt)) 
    #     print("\nOptimal solution: {}\n".format(pulp.value(problem.objective)))
    #     print("Decision variables:")
    #     for var in problem.variables() :
    #         print("{} = {}".format(var.name, var.varValue))

    # return True

def feasibilityCheck(data, period) :
    "Check basic EDF feasibility on one target"
    for taskSet in data.values() :
        taskWCETs = list(map(lambda x: x.WCET, taskSet))
        if sum(taskWCETs) > period :
            return False

    return True

def getPeriodBounds(data) :
    "Returns period bounds to be used on the optimization problem, depending on user preferences"
    min = 0
    max = 0
    for taskSet in data.values() :
        taskTimes = list(map(lambda x: x.WCET + x.Tx + x.slotBuff, taskSet))
        sumTimes = sum(taskTimes)
        if sumTimes > min :
            min = sumTimes
        max = max + sumTimes
    
    return (min, max)

class TimeUtilityFunction():
    "Time Utility Functions (TUF) class"
    epsilon = 0.1
    def __init__(self, ID, wcet, period, weight) :
        self.ID = ID
        self.wcet = wcet
        self.period = period
        self.weight = 4 if weight else 1

    def offset(self, x) :
        alpha = 0
        beta = self.period - self.wcet
        return self.weight*((1/(beta - alpha))*((self.epsilon - 1)*x + beta - alpha*self.epsilon))

    def deadline(self, x) :
        alpha = self.wcet
        beta = self.period
        return self.weight*((1/(beta - alpha))*((self.epsilon - 1)*x + beta - alpha*self.epsilon))

def formulateDependencies(data, baudRate, timeUnit, slotBuff) :
    targetIDGroups = itertools.groupby(sorted(data, key = lambda x: x.targetID), lambda x: x.targetID)
    targetIDSets = {}
    for key, group in targetIDGroups: 
        targetIDSets[key] = list(group)
    
    transformedData = {ID:[] for ID in targetIDSets.keys()}

    for target, taskSet in targetIDSets.items() :
        taskIDs = list(map(lambda x: x.taskID, taskSet))
        for task in taskSet :
            isPayload = False
            payload = []
            interDependencies = []
            consumerDependencies = []
            producerDependencies = []
            for param in task.inputs :
                if param[1] in taskIDs :
                    #there's an inter dependency, should be scheduled after taskID: param
                    interDependencies.append(param)
                else :
                    #there's a communication consumer dependency, should be scheduled after taskID: param..i.e alpha bound tic
                    consumerDependencies.append(param)
            for param in task.outputs :
                for val in param[1] :
                    if val not in taskIDs : 
                        #there's a communication producer dependency, should be scheduled before taskID: val..i.e beta bound tip
                        isPayload = True
                        producerDependencies.append(val)
                if isPayload :
                    payload.append(param[0])
                    isPayload = False
            
            Tx = computeTxTime(baudRate, timeUnit, payload)
            buffer = slotBuff if producerDependencies != [] else 0
            transformedData[target].append(taskItemTransformation(task.taskID, task.WCET, Tx, buffer, interDependencies, consumerDependencies, producerDependencies))

    return transformedData

def extractSlots(data, lastSlotBuff, period, overlapBound) :
    producersTasks = []
    for target, taskSet in data.items() :
        producersTasks.extend(list(map(lambda x: (x.offset, x.deadline + x.Tx, x.slotBuff), filter(lambda x: x.taskType & ProducerTask != 0, taskSet))))
    producersTasks = sorted(producersTasks, key = lambda x: x[0])
    
    timeSlots = {}
    slotStart = 0
    for index, param in enumerate(producersTasks) :
        timeSlots[index] = timeSlot(slotStart, slotStart + overlapBound, param[1], param[1] + param[2], [])
        slotStart = param[1] + param[2]
    
    timeSlots[len(producersTasks)] = timeSlot(slotStart, slotStart + overlapBound, period, period + lastSlotBuff, [])
    
    for target, taskSet in data.items() :
        for task in taskSet :
            for slot in timeSlots.values() :
                if task.offset in range(int(slot.start), int(slot.end)) :
                    slot.tasks.append(timeSlotTask(target, task.taskID, task.offset, task.deadline, task.WCET, task.Tx, task.taskType))
                    break

    return timeSlots

def fixedSlotComputations(data, slots) :
    slotDiff = {}
    maxSlot = max(list(map(lambda x: x.end - x.start, slots.values())))
    for slotID, slot in slots.items() :
        slotDiff[slotID] = maxSlot - (slot.end - slot.start)

    transformedData = {ID:[] for ID in data.keys()}

    for target, taskSet in data.items() :
        for task in taskSet :
            if task.producerDependencies != [] :
                for slotID, slot in slots.items() :
                    slotTaskIDs = list(map(lambda x: x.taskID, slot.tasks))
                    if task.taskID in slotTaskIDs :
                        break 
                modifiedTask = task._replace(slotBuff = task.slotBuff + slotDiff[slotID])
                transformedData[target].append(modifiedTask)
            else :    
                transformedData[target].append(task)

    return (transformedData, slotDiff[len(slots) - 1])        

def computeTxTime(baudRate, timeUnit, payload) :
    if not payload :
        return 0

    #worst case number of bits including bit stuffing
    frameBits = lambda n : 8*n + 44 + (34 + 8*n - 1)/4 
    
    payloadBytes = 0
    for param in payload :
        if param == "int" :
            payloadBytes = payloadBytes + intBytes
        elif param == "float" :
            payloadBytes = payloadBytes + floatBytes
        else :
            payloadBytes = payloadBytes + boolBytes

    fullPayloadFrames = int(payloadBytes/8)
    leftPayload = payloadBytes % 8
    totalBits = fullPayloadFrames*frameBits(8) + frameBits(leftPayload) if leftPayload != 0 else fullPayloadFrames*frameBits(8)

    if baudRate == "125 Kbps" :
        txSeconds = totalBits/125e3
    elif baudRate == "250 Kbps" :
        txSeconds = totalBits/250e3
    elif baudRate == "500 Kbps" :
        txSeconds = totalBits/500e3
    else :
        txSeconds = totalBits/1e6

    if timeUnit == "ns" :
        txTime = txSeconds*1e9
    elif timeUnit == "µs" :
        txTime = txSeconds*1e6
    elif timeUnit == "ms" :
        txTime = txSeconds*1e3
    else :
        txTime = txSeconds

    return round(txTime)

def convertTime(value, timeUnit) :
    sec = 0
    nsec = 0
    if timeUnit == "ns" :
        nsec = value
    elif timeUnit == "µs" :
        nsec = value*1e3
    elif timeUnit == "ms" :
        nsec = value*1e6
    else :
        sec = value
    
    return (int(sec), int(nsec))

def gitVersion():
    logging.getLogger(__name__).debug("Reading application's version from '{}version.txt'...".format(workingDirectory))
    try :
        with open(workingDirectory+"version.txt", "r") as inFile :
            version = inFile.read().strip("\n")
            return version
    except :
        logging.getLogger(__name__).critical("While reading the version file, UNEXPECTED exception occurred!\n\n", exc_info = True)
        return "null"

class targetsDataClass :
    "Targets data class"
    def __init__(self) :
        self.targetsData = {}

    def fetchTargetsData(self) :
        targetsFiles = list(pathlib.Path(targetsPath).glob("*.json"))
        logging.getLogger(__name__).warning("Fetching targets data from {}...".format(targetsFiles))
        if targetsFiles == [] :
            logging.getLogger(__name__).warning("There are NO target's config files (.json) in {}!".format(targetsPath))
            return False
        else :
            for file in targetsFiles :
                try :
                    with open(file, "r", encoding = "utf-8") as f :
                        data = json.load(f)
                        if self.targetConfigSchemaCheck(data, file) :
                            self.targetsData[data["name"]] = targetClass(data)
                        else :
                            logging.getLogger(__name__).error("Could NOT fetch target's config file {}!".format(file))
                except json.JSONDecodeError as err :
                    logging.getLogger(__name__).error("While fetching target's config file, JSON decoder failed!\n\n{} in {}.\n".format(str(err), file))
                except :
                    logging.getLogger(__name__).critical("While fetching target's config file, UNEXPECTED exception occurred!\n\n", exc_info = True)
            if self.targetsData == {} :
                logging.getLogger(__name__).warning("NO target's config file were fetched!")
                return False
            else :
                logging.getLogger(__name__).warning("Fetched targets: {}.".format(list(self.targetsData.keys())))
                return True

    def targetConfigSchemaCheck(self, config, file) :
        try :
            with open(schemaPath+"target_config_schema.json", "r", encoding = "utf-8") as outFile :
                schema = json.load(outFile)
        except json.JSONDecodeError as err :
            logging.getLogger(__name__).error("While loading target config schema, JSON decoder failed!\n\n{} in {}.\n".format(str(err), schemaPath+"target_config_schema.json"))
            return False
        except :
            logging.getLogger(__name__).critical("While loading target config schema, UNEXPECTED exception occurred!\n\n", exc_info = True)
            return False
        try :
            jsonschema.validate(config, schema)
        except jsonschema.SchemaError as err :
            logging.getLogger(__name__).error("While loading target config schema!\n\n{}\n".format(err))
            return False
        except jsonschema.ValidationError as err :
            instance = "On instance"
            for item in err.path :
                instance = instance+"[{}]".format(item) if type(item) == int else instance+"['{}']".format(item)
            logging.getLogger(__name__).warning("Target config {} doesn't meet schema requirements!\n\n{}.\n\n{}:\n{}\n".format(file, err.message, instance, json.dumps(err.instance, indent = 2, ensure_ascii = False)))
            return False
        else :
            logging.getLogger(__name__).warning("Target config {} schema check was successfully passed...".format(file))
            return True 

    def targetHwSwMatch(self, name, hw, sw) :
        logging.getLogger(__name__).debug("Checking target SW/HW match...")
        if not name in self.targetsData :
            errorMessage = "'{}' is not a recognized target.".format(name)
            return (False, errorMessage)
        target = self.targetsData[name]
        if target.hw != hw :
            errorMessage = "HW version: {} of '{}' target doesn't match.".format(target.hw, name)
            return (False, errorMessage)
        if target.sw != sw :
            errorMessage = "SW version: {} of '{}' target doesn't match.".format(target.sw, name)
            return (False, errorMessage)

        return (True, "")

    def targetTaskCheck(self, taskName, targetName) :
        logging.getLogger(__name__).debug("Target task check called...")
        target = self.targetsData[targetName]
        task = target.hasTask(taskName)
        if not task :
            errorMessage = "'{}' is not a task of '{}' target.".format(taskName, targetName)
            return (False, [], [], errorMessage)
        
        return (True, task.inputs, task.outputs, "")

def scheduleScriptCheck(file) :
    try :
        with open(file, "r", encoding = "utf-8") as outFile :
            data = json.load(outFile)
    except json.JSONDecodeError as err :
        logging.getLogger(__name__).error("While reading schedule script, JSON decoder failed!\n\n{} in {}.\n".format(str(err), file))
        return False
    except FileNotFoundError:
        raise FileNotFoundError
        # logging.getLogger(__name__).critical("While reading schedule script, UNEXPECTED exception occurred!\n\n", exc_info = True)
        return False
    
    if scheduleScriptSchemaCheck(data, file.replace(".json", "")) :
        if scheduleScriptSanityCheck(data, file.replace(".json", "")) :
            logging.getLogger(__name__).warning("Schedule script complete-check for {} was SUCCESSFULLY passed.".format(file.replace(".json", "")))
            return True
        else :
            logging.getLogger(__name__).error("While checking schedule script, sanity check was NOT passed!")
            return False
    else :
        logging.getLogger(__name__).error("While checking schedule script, schema check was NOT passed!")
        return False

def scheduleScriptSchemaCheck(data, file) :
    try :
        with open(schemaPath+"schedule_script_schema.json", "r", encoding = "utf-8") as outFile :
            schema = json.load(outFile)
    except json.JSONDecodeError as err :
        logging.getLogger(__name__).error("While loading schedule script schema, JSON decoder failed!\n\n{} in {}.\n".format(str(err), schemaPath+"schedule_script_schema.json"))
        return False
    except :
        logging.getLogger(__name__).critical("While loading schedule scripts schema, UNEXPECTED exception occurred!\n\n", exc_info = True)
        return False

    try :
        jsonschema.validate(data, schema)
    except jsonschema.SchemaError as err :
        logging.getLogger(__name__).error("While loading schedule script schema!\n\n{}\n".format(err))
        return False
    except jsonschema.ValidationError as err :
        instance = "On instance"
        for item in err.path :
            instance = instance+"[{}]".format(item) if type(item) == int else instance+"['{}']".format(item)
        logging.getLogger(__name__).warning("Schedule script {} doesn't meet schema requirements!\n\n{}.\n\n{}:\n{}\n".format(file, err.message, instance, json.dumps(err.instance, indent = 2, ensure_ascii = False)))
        return False
    else :
        logging.getLogger(__name__).warning("Schedule script {} schema check was successfully passed...".format(file))
        return True  

def scheduleScriptSanityCheck(data, file) :
    errorMessage = "Schedule script file %s doesn't meet sanity requirements!\n\n{}\n\n{}:\n{}\n" % file   
    targetsObject = targetsDataClass()
    if not targetsObject.fetchTargetsData() :
        return False

    targets = data["targets"]
    canIDList = []
    for index, item in enumerate(targets) :
        ret = targetsObject.targetHwSwMatch(item["name"], item["hw"], item["sw"])
        if not ret[0] :
            logging.getLogger(__name__).warning(errorMessage.format(ret[1], "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
            return False

        canID = item["can id"] 
        if canID in canIDList :
            logging.getLogger(__name__).warning(errorMessage.format("'can id' value already assigned for other target, it should be unique.", "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
            return False
        canIDList.append(canID)

    variables = data["variables"]
    variablesCount = len(variables)
    usedVariables = set()

    tasks = data["tasks"]
    targetsCount = len(targets)
    for index, item in enumerate(tasks) :
        targetIndex = item["target"]
        if targetIndex >= targetsCount : 
            logging.getLogger(__name__).warning(errorMessage.format("'target' value should be smaller than 'targets' count.", "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
            return False
        
        taskName = item["name"]
        ret = targetsObject.targetTaskCheck(taskName, targets[targetIndex]["name"]) 
        if not ret[0] :
            logging.getLogger(__name__).warning(errorMessage.format(ret[3], "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
            return False
        
        if ret[1] :
            inputs = ret[1]
            inputsValues = item["in"]
            if len(inputs) != len(inputsValues) :
                logging.getLogger(__name__).warning(errorMessage.format("'{}' expects [{}] inputs.".format(taskName, len(inputs)), "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

            for i, param in enumerate(inputs) :
                if inputsValues[i] >= variablesCount : 
                    logging.getLogger(__name__).warning(errorMessage.format("'in[{}]' value should be smaller than 'variables' count.".format(i), "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                    return False

                usedVariables.add(inputsValues[i])                
                valueType = variables[inputsValues[i]]
                if param.type != valueType :
                    logging.getLogger(__name__).warning(errorMessage.format("'variables[in[{}]]' value should be of type '{}'.".format(i, param.type), "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                    return False

        if ret[2] :
            outputs = ret[2]
            outputsValues = item["out"]
            if len(outputs) != len(outputsValues) :
                logging.getLogger(__name__).warning(errorMessage.format("'{}' expects [{}] outputs.".format(taskName, len(outputs)), "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

            for i, param in enumerate(outputs) :
                if outputsValues[i] >= variablesCount : 
                    logging.getLogger(__name__).warning(errorMessage.format("'out[{}]' value should be smaller than 'variables' count.".format(i), "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                    return False

                usedVariables.add(outputsValues[i])                
                valueType = variables[outputsValues[i]]
                if param.type != valueType :
                    logging.getLogger(__name__).warning(errorMessage.format("'variables[out[{}]]' value should be of type '{}'.".format(i, param.type), "On instance['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                    return False
    
    if usedVariables != set(range(variablesCount)) :
        logging.getLogger(__name__).warning(errorMessage.format("'variables' should all be used, they are in excess.", "On instance['variable']", json.dumps(variables, indent = 2, ensure_ascii = False)))
        return False

    round = data["round"]
    period = (round["period"]["sec"], round["period"]["nsec"])

    slots = round["slots"]
    tasksCount = len(tasks)
    for index, item in enumerate(slots) :
        assignedTarget = item["assign"]["target"]
        assignedTasks = item["assign"]["tasks"]

        if assignedTarget >= targetsCount : 
            logging.getLogger(__name__).warning(errorMessage.format("'target' value should be smaller than 'targets' count.", "On instance['round']['slots'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
            return False

        for elt in assignedTasks :
            if elt >= tasksCount : 
                logging.getLogger(__name__).warning(errorMessage.format("'tasks' values should be smaller than 'tasks' count.", "On instance['round']['slots'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

            if tasks[elt]["target"] != assignedTarget :
                logging.getLogger(__name__).warning(errorMessage.format("'tasks[{}]' which is an assigned task, doesn't belong to the assigned target.".format(elt), "On instance['round']['slots'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

    lastSlotEnd = (item["time"]["end"]["sec"], item["time"]["end"]["nsec"])
    if lastSlotEnd != period :
        logging.getLogger(__name__).warning(errorMessage.format("'end' of last slot doesn't match period [{}].".format(period), "On instance['round']['slots'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
        return False

    logging.getLogger(__name__).warning("Schedule script {} sanity check was successfully passed...".format(file))    
    return True


class optimizationThread(QThread) :
    "Optimization thread class"
    optDone = pyqtSignal(tuple)
    def __init__(self, parent, data, baudRate, timeUnit, slotBuff, slotRoutine, slotOverlapBound) :
        super().__init__(parent)
        
        self.data = data
        self.baudRate = baudRate
        self.timeUnit = timeUnit
        self.slotBuff = slotBuff
        self.slotRoutine = slotRoutine
        self.slotOverlapBound = slotOverlapBound
    
    def run(self) :
        logging.getLogger(__name__).debug("Generating schedule from: {}...".format(self.data))
        formulatedData = formulateDependencies(self.data, self.baudRate, self.timeUnit, self.slotBuff)
        logging.getLogger(__name__).debug("Formulated data: {}...".format(formulatedData))
        periodBounds = getPeriodBounds(formulatedData)
        logging.getLogger(__name__).debug("Period bounds: {}...".format(periodBounds))
        lastSlotBuff = self.slotBuff

        if self.slotRoutine == "Fixed" :
            problem = modeling(formulatedData, periodBounds[1])
            ret = optimize(problem, formulatedData)
            if not ret[0] :
                logging.getLogger(__name__).warning("The scheduling problem is UNFEASIBLE!")
                logging.getLogger(__name__).debug("Formulated data: {}".format(formulatedData))
                self.optDone.emit((False, "", "", "", ""))
                return 

            timeSlots = extractSlots(ret[1], lastSlotBuff, ret[2], self.slotOverlapBound)
            ret = fixedSlotComputations(formulatedData, timeSlots)
            formulatedData = ret[0]
            periodBounds = getPeriodBounds(formulatedData)
            lastSlotBuff = lastSlotBuff + ret[1]

        problem = modeling(formulatedData, periodBounds[1])
        ret = optimize(problem, formulatedData)
        if not ret[0] :
            logging.getLogger(__name__).warning("The scheduling problem is UNFEASIBLE!")
            logging.getLogger(__name__).debug("Formulated data: {}".format(formulatedData))
            self.optDone.emit((False, "", "", "", ""))
            return  

        timeSlots = extractSlots(ret[1], lastSlotBuff, ret[2], self.slotOverlapBound)
        self.optDone.emit((True, ret[1], ret[2] + lastSlotBuff, timeSlots, formulatedData))

#----------------------------------------------------------------------------#