############################################
##      Filename:     main_window.py      ##
##      Date:         18/04/2020          ##
##      Author:       Ayman HATOUM        ##
############################################
"""This file contains the class definition of mainWindow(). A subclass
of QMainWindow() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import json
import jsonschema
import logging
import logging.config
from PyQt5.QtCore import Qt
from PyQt5.QtCore import QSettings
from PyQt5.QtCore import QEvent
from PyQt5.QtWidgets import QMainWindow
from PyQt5.QtWidgets import QDesktopWidget
from PyQt5.QtWidgets import QGraphicsView
from PyQt5.QtWidgets import QGraphicsItem
from PyQt5.QtWidgets import QFileDialog
from PyQt5.QtWidgets import QMessageBox
from PyQt5.QtWidgets import QWhatsThis
from PyQt5.QtGui import QIcon
from PyQt5.QtGui import QGuiApplication
from src import tools
from src.constants import picturesPath
from src.constants import displayHeight
from src.constants import tdcsIcon
from src.constants import displayWidth
from src.constants import colorNames
from src.constants import schemaPath
from src.constants import logPath
from src.constants import startupMsg
from src.constants import workingDirectory
from src.main_actions import mainActions
from src.bars.menu_bar import menuBar
from src.bars.tool_bar import toolBar
from src.bars.status_bar import statusBar
from src.windows.targets.targets_window import targetsWindow
from src.windows.prompt.prompt_window import promptWindow
from src.windows.central.central_window import centralWindow
from src.dialogs.new_target_dialog import newTargetDialog
from src.dialogs.save_dialog import saveDialog
from src.dialogs.preference_settings_dialog import preferenceSettingsDialog
from src.dialogs.display_settings_dialog import displaySettingsDialog
from src.dialogs.about_dialog import aboutDialog
from src.dialogs.help_contents_dialog import helpContentsDialog

#----------------------------------------------------------------------------#

class mainWindow(QMainWindow) :
    "Main window class"
    def __init__(self):
        super().__init__()
        
        self.currentProjectName = "untitled"
        self.currentProjectSaved = False
        self.setWindowTitle(self.currentProjectName+".tdcs[*]")
        self.setWindowIcon(QIcon(picturesPath+tdcsIcon))
        
        self.built = True

        #Necessary for streaming log
        self.prompt = promptWindow(self)

        self.configureLogger()
        self.version = tools.gitVersion()
        logging.getLogger(__name__).info("Git version: '{}'.".format(self.version))
        self.createLayout()
        self.readSettings()

    def configureLogger(self) :
        logging.logFile = workingDirectory+"log.tdcs.log"
        logging.promptStream = self.prompt.terminal.logView
        logging.config.fileConfig(logPath+"logging.conf")

        handlers = logging.getLogger().handlers
        simpleFormater = handlers[0].formatter
        detailedFormater = handlers[1].formatter
        for handler in handlers :
            handler.setFormatter(logging.Formatter("%(message)s"))
    
        logging.getLogger(__name__).info(startupMsg)
        
        handlers[0].setFormatter(simpleFormater) 
        handlers[1].setFormatter(detailedFormater)

        logging.getLogger(__name__).debug("Logging configured...")

    def checkLogDebug(self, checked) :
        logging.getLogger(__name__).debug("Log debug action triggered...")
        if checked :
            fileHandler = logging.getLogger().handlers[2]
            fileHandler.setLevel(logging.DEBUG)
        else :
            fileHandler = logging.getLogger().handlers[2]
            fileHandler.setLevel(logging.INFO)

    def readSettings(self) :
        logging.getLogger(__name__).debug("Reading settings from windows registry...")
        settings = QSettings()
        geometry = settings.value("geometry")
        frame = settings.value("frame")
        if geometry and frame :
            self.resize(geometry.size())
            self.move(frame.topLeft())
        else :
            self.setDisplay()
        maximized = settings.value("maximized")
        if maximized :
            self.setWindowState(Qt.WindowMaximized)

    def createLayout(self) :
        logging.getLogger(__name__).debug("Creating layout and all widgets...")
        #Setting app actions
        self.allActions = mainActions(self)
        
        #Setting menu bar
        self.menuBar_ = menuBar(self, self.allActions)
        self.setMenuBar(self.menuBar_)
        
        #Setting main tool bar
        self.toolBar_ = toolBar(self, self.allActions)
        self.addToolBar(self.toolBar_)
        
        #Setting status bar
        self.statusBar_ = statusBar(self)
        self.setStatusBar(self.statusBar_)
        
        #Setting widgets
        self.targets = targetsWindow(self)
        self.central = centralWindow(self)
        self.setCentralWidget(self.central)
        
        self.addDockWidget(Qt.LeftDockWidgetArea, self.targets)
        self.addDockWidget(Qt.BottomDockWidgetArea, self.prompt)
        
        #Setting options
        self.setAnimated(False)                                 #True will smooth the adjusting content (docks & toolbars)
        # self.setDockOptions(QMainWindow.ForceTabbedDocks)     #if want them to be only tabified dock widgets
                
        #Setting slots
        self.central.builder.scene.selectionChanged.connect(self.sceneSelectionChanged)
        self.central.builder.scene.rebuildDesign.connect(self.builtChanged)
        self.targets.tree.modifiedCanID.connect(self.builtChanged)
        self.central.currentChanged.connect(self.centralTabChanged)
        self.central.scheduler.generateFinish.connect(self.generateScheduleFinish)
    
    def setDisplay(self) :
        "Sets the window display position and size depending on the available desktop geometry"
        logging.getLogger(__name__).debug("Setting display with specified desktop ratios...")
        desktopGeometry = QDesktopWidget().availableGeometry()
        self.resize(int(desktopGeometry.width()*displayWidth), int(desktopGeometry.height()*displayHeight))
        frame = self.frameGeometry()
        frame.moveCenter(desktopGeometry.center())
        self.move(frame.topLeft())

    def showDisplaySettingsDialog(self) :
        logging.getLogger(__name__).debug("Display settings action triggered...")
        dialog = displaySettingsDialog(self)
        if dialog.exec() :
            logging.getLogger(__name__).debug("Display settings dialog accepted: {}...".format(dialog.data))
            self.central.scheduler.updateDisplaySettings(dialog.data)
        else :
            logging.getLogger(__name__).debug("Display settings dialog canceled...")

    def showPrefSettingsDialog(self) :
        logging.getLogger(__name__).debug("Preference settings action triggered...")
        dialog = preferenceSettingsDialog(self)
        if dialog.exec() :
            logging.getLogger(__name__).debug("Preference settings dialog accepted: {}...".format(dialog.data))
            self.central.scheduler.updateSettings(dialog.data)
            self.builtChanged()
        else :
            logging.getLogger(__name__).debug("Preference settings dialog canceled...")

    def generateSchedule(self) :
        logging.getLogger(__name__).debug("Generate action triggered...")
        if self.isWindowModified() :
            if not self.save() :
                return
        #Check for builder design sanity & extract data
        logging.getLogger(__name__).info("Building design...")
        QGuiApplication.setOverrideCursor(Qt.WaitCursor)
        sanity = self.central.builder.scene.builderSanityCheck()
        if sanity :
            logging.getLogger(__name__).info("Building design was SUCCESSFULLY completed.")
            data = self.central.builder.scene.extractBuilderData()
        else :
            logging.getLogger(__name__).error("While building design, builder sanity check was NOT passed!")
            QGuiApplication.restoreOverrideCursor()
            return
        QGuiApplication.restoreOverrideCursor()

        #Formulate data & model optimization problem
        logging.getLogger(__name__).info("Generating schedule...")
        self.central.scheduler.generateSchedule(data)
             
    def generateScheduleFinish(self, ret) :
        if not ret[0] :
            logging.getLogger(__name__).error("Schedule CANNOT be generated with the specified parameters!")
            return
            
        self.built = True
        self.allActions.generateAction.setEnabled(False)
        self.allActions.exportScheduleAction.setEnabled(True)
        self.allActions.exportScheduleDescriptionAction.setEnabled(True)
        logging.getLogger(__name__).info("Generating schedule was SUCCESSFULLY completed. Period: [{}]".format(ret[1]))
        self.central.setCurrentIndex(1) 

    def exportScheduleDescription(self) :
        logging.getLogger(__name__).debug("Showing save as dialog for exporting schedule description...")
        exportFile = QFileDialog.getSaveFileName(self, "Save As", self.currentProjectName, "SD files (*.sd.json)")
        if exportFile[0] :
            QGuiApplication.setOverrideCursor(Qt.WaitCursor)
            logging.getLogger(__name__).debug("Exporting schedule description into {}...".format(exportFile[0]))
            try :
                with open(exportFile[0], "w", encoding = "utf-8") as outFile :
                    json.dump(self.central.scheduler.extractedScheduleDescription, outFile, indent = 4, ensure_ascii = False)
            except :
                logging.getLogger(__name__).critical("While exporting schedule description, UNEXPECTED exception occurred!\n\n", exc_info = True)
                QGuiApplication.restoreOverrideCursor()
                return     
        else :
            logging.getLogger(__name__).debug("Save as dialog canceled...")
            return   
        
        logging.getLogger(__name__).info("Schedule description saved to {}.".format(exportFile[0].replace(".json", "")))
        QGuiApplication.restoreOverrideCursor()

    def exportSchedule(self) :
        logging.getLogger(__name__).debug("Showing save as dialog for exporting schedule script...")
        exportFile = QFileDialog.getSaveFileName(self, "Save As", self.currentProjectName, "SS files (*.ss.json)")
        if exportFile[0] :
            QGuiApplication.setOverrideCursor(Qt.WaitCursor)
            logging.getLogger(__name__).debug("Exporting schedule script into {}...".format(exportFile[0]))
            try :
                with open(exportFile[0], "w", encoding = "utf-8") as outFile :
                    json.dump(self.central.scheduler.extractedSchedule, outFile, indent = 4, ensure_ascii = False)
            except :
                logging.getLogger(__name__).critical("While exporting schedule script, UNEXPECTED exception occurred!\n\n", exc_info = True)
                QGuiApplication.restoreOverrideCursor()
                return     
        else :
            logging.getLogger(__name__).debug("Save as dialog canceled...")
            return   
        
        logging.getLogger(__name__).info("Schedule script saved to {}.".format(exportFile[0].replace(".json", "")))
        QGuiApplication.restoreOverrideCursor()

    def newProject(self) :
        logging.getLogger(__name__).debug("New action triggered...")
        if not self.checkForSave() :
            return
        self.clearProject()
        self.targets.fetchTargetsData()
        self.currentProjectName = "untitled"
        self.currentProjectSaved = False
        self.setWindowTitle(self.currentProjectName+".tdcs[*]")
        self.allActions.newAction.setEnabled(False)
        self.built = True
        self.allActions.generateAction.setEnabled(False)
        self.central.scheduler.restart()
        logging.getLogger(__name__).info("New project {} opened.".format(self.currentProjectName+".tdcs"))

    def clearProject(self) :
        logging.getLogger(__name__).debug("Clearing project...")
        self.central.builder.scene.clearScene()
        self.targets.resetTargets()
        self.setWindowModified(False)
        self.allActions.exportScheduleAction.setEnabled(False)
        self.allActions.exportScheduleDescriptionAction.setEnabled(False)
        self.central.setCurrentIndex(0)    

    def showOpenDialog(self) :
        logging.getLogger(__name__).debug("Open action triggered...")
        if not self.checkForSave() :
            return
        openFile = QFileDialog.getOpenFileName(self, "Open Project", "", "TDCS files (*.tdcs.json)")
        if openFile[0] :
            if self.loadDescriptionFile(openFile[0]) :
                self.currentProjectName = openFile[0].replace(".tdcs.json", "")
                self.currentProjectSaved = True
                self.setWindowTitle(self.currentProjectName+".tdcs[*]")
        else :
            logging.getLogger(__name__).debug("Open dialog canceled...")

    def loadDescriptionFile(self, file) :
        logging.getLogger(__name__).debug("Loading description file from {}...".format(file))
        QGuiApplication.setOverrideCursor(Qt.WaitCursor)
        load = False
        try :
            with open(file, "r", encoding = "utf-8") as outFile :
                data = json.load(outFile)
        except json.JSONDecodeError as err :
            logging.getLogger(__name__).error("While loading description file, JSON decoder failed!\n\n{} in {}.\n".format(str(err), file))
            QGuiApplication.restoreOverrideCursor()
            return load
        except :
            logging.getLogger(__name__).critical("While loading description file, UNEXPECTED exception occurred!\n\n", exc_info = True)
            QGuiApplication.restoreOverrideCursor()
            return load
        
        if self.descriptionSchemaCheck(data, file.replace(".json", "")) :
            if self.descriptionSanityCheck(data, file.replace(".json", "")) :
                self.clearProject()
                self.central.scheduler.restart()
                self.targets.loadTargets(data["targets"])
                self.central.builder.scene.loadScene(data["scene"])
                self.central.scheduler.loadSettings(data["preferred settings"])
                self.setWindowModified(False)
                self.builtChanged()
                load = True
                logging.getLogger(__name__).info("Project {} was SUCCESSFULLY loaded.".format(file.replace(".json", "")))
            else :
                logging.getLogger(__name__).error("While loading description file, sanity check was NOT passed!")
        else :
            logging.getLogger(__name__).error("While loading description file, schema check was NOT passed!")
        QGuiApplication.restoreOverrideCursor()
        return load

    def descriptionSchemaCheck(self, description, file) :
        try :
            with open(schemaPath+"description_file_schema.json", "r", encoding = "utf-8") as outFile :
                schema = json.load(outFile)
        except json.JSONDecodeError as err :
            logging.getLogger(__name__).error("While loading description file schema, JSON decoder failed!\n\n{} in {}.\n".format(str(err), schemaPath+"description_file_schema.json"))
            return False
        except :
            logging.getLogger(__name__).critical("While loading description file schema, UNEXPECTED exception occurred!\n\n", exc_info = True)
            return False

        try :
            jsonschema.validate(description, schema)
        except jsonschema.SchemaError as err :
            logging.getLogger(__name__).error("While loading description file schema!\n\n{}\n".format(err))
            return False
        except jsonschema.ValidationError as err :
            instance = "On instance"
            for item in err.path :
                instance = instance+"[{}]".format(item) if type(item) == int else instance+"['{}']".format(item)
            logging.getLogger(__name__).warning("Description file {} doesn't meet schema requirements!\n\n{}.\n\n{}:\n{}\n".format(file, err.message, instance, json.dumps(err.instance, indent = 2, ensure_ascii = False)))
            return False
        else :
            logging.getLogger(__name__).debug("Description file schema check was successfully passed...")
            return True        

    def descriptionSanityCheck(self, description, file) :
        errorMessage = "Description file %s doesn't meet sanity requirements!\n\n{}\n\n{}:\n{}\n" % file   

        targets = description["targets"]
        labelList = []
        canIDList = []
        colorList = []
        for index, item in enumerate(targets) :
            ret = self.targets.targetHwSwMatch(item["name"], item["hw"], item["sw"])
            if not ret[0] :
                logging.getLogger(__name__).warning(errorMessage.format(ret[1], "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            
            label = item["label"] 
            if label in labelList :
                logging.getLogger(__name__).warning(errorMessage.format("'label' value already assigned for other target, it should be unique.", "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            labelList.append(label)

            canID = item["can id"] 
            if canID in canIDList :
                logging.getLogger(__name__).warning(errorMessage.format("'can id' value already assigned for other target, it should be unique.", "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            canIDList.append(canID)

            color = item["color"] 
            if not color in colorNames :
                logging.getLogger(__name__).warning(errorMessage.format("'color' value should be one of the predefined colors.", "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            if color in colorList :
                logging.getLogger(__name__).warning(errorMessage.format("'color' value already assigned for other target, it should be unique.", "On instance['targets'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            colorList.append(color)

        sceneDescription = description["scene"]
               
        tasks = sceneDescription["tasks"]
        targetsCount = len(targets)
        tasksIDList = []
        for index, item in enumerate(tasks) :
            targetIndex = item["target index"]
            if targetIndex >= targetsCount : 
                logging.getLogger(__name__).warning(errorMessage.format("'target index' value should be smaller than 'targets' count.", "On instance['scene']['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

            taskName = item["name"]
            ret = self.targets.targetTaskCheck(taskName, targets[targetIndex]["name"]) 
            if not ret[0] :
                logging.getLogger(__name__).warning(errorMessage.format(ret[3], "On instance['scene']['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            if ret[1] :
                inputs = ret[1]
                try :
                    defaultInputs = item["default inputs"]
                except KeyError :
                    logging.getLogger(__name__).warning(errorMessage.format("'{}' task has inputs, 'default inputs' property expected.".format(taskName), "On instance['scene']['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                    return False

                for param in inputs :
                    if not param.label in defaultInputs :            
                        logging.getLogger(__name__).warning(errorMessage.format("'{}' input of task '{}' is expected as an entry.".format(param.label, taskName), "On instance['scene']['tasks'][{}]['default inputs']".format(index), json.dumps(defaultInputs, indent = 2, ensure_ascii = False)))
                        return False
                    
                    val = param.valueValid(defaultInputs[param.label])
                    if not val[0] :
                        logging.getLogger(__name__).warning(errorMessage.format("'{}' input of task '{}' {}".format(param.label, taskName, val[1]), "On instance['scene']['tasks'][{}]['default inputs']".format(index), json.dumps(defaultInputs, indent = 2, ensure_ascii = False)))
                        return False

            taskID = item["id"]
            if taskID in tasksIDList :
                logging.getLogger(__name__).warning(errorMessage.format("'id' value already assigned for other target, it should be unique.", "On instance['scene']['tasks'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            tasksIDList.append(taskID)

        connections = sceneDescription["connections"]
        inputsConnected = []
        for index, item in enumerate(connections) :
            outtask = item["outtask"]
            output = item["output"]

            ret = (False, [], [])
            for taskItem in tasks :
                if taskItem["id"] == outtask :
                    ret = self.targets.targetTaskCheck(taskItem["name"], targets[taskItem["target index"]]["name"]) 
                    break
            if not ret[0] :
                logging.getLogger(__name__).warning(errorMessage.format("'outtask' value should be an already existing ID of 'task' items.", "On instance['scene']['connections'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            
            val = False
            for param in ret[2] :
                if param.label == output :
                    val = True
                    break
            if not val :
                logging.getLogger(__name__).warning(errorMessage.format("'{}' is not an output of '{}'.".format(output, taskItem["name"]), "On instance['scene']['connections'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

            intask = item["intask"]
            input_ = item["input"]

            ret = (False, [], [])
            for taskItem in tasks :
                if taskItem["id"] == intask :
                    ret = self.targets.targetTaskCheck(taskItem["name"], targets[taskItem["target index"]]["name"]) 
                    break
            if not ret[0] :
                logging.getLogger(__name__).warning(errorMessage.format("'intask' value should be an already existing ID of 'task' items.", "On instance['scene']['connections'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            
            val = False
            for param in ret[1] :
                if param.label == input_ :
                    val = True
                    break
            if not val :
                logging.getLogger(__name__).warning(errorMessage.format("'{}' is not an input of '{}'.".format(input_, taskItem["name"]), "On instance['scene']['connections'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False

            if (intask, input_) in inputsConnected :
                logging.getLogger(__name__).warning(errorMessage.format("Input pointed to by ({}, {}) is already connected to other output.".format(intask, input_), "On instance['scene']['connections'][{}]".format(index), json.dumps(item, indent = 2, ensure_ascii = False)))
                return False
            inputsConnected.append((intask, input_))

        logging.getLogger(__name__).debug("Description file sanity check was successfully passed...")
        return True

    def checkForSave(self) :
        logging.getLogger(__name__).debug("Checking for save...")
        proceed = True
        if self.isWindowModified() :
            dialog = saveDialog(self)
            ret = dialog.exec()
            if ret == QMessageBox.Save :
                proceed = self.save()
            elif ret == QMessageBox.Discard :
                logging.getLogger(__name__).debug("Save option discarded...")
                pass
            else :
                logging.getLogger(__name__).debug("Save option canceled...")
                proceed = False
        return proceed

    def save(self) :
        logging.getLogger(__name__).debug("Save function called...")
        if self.currentProjectSaved :
            saved = self.saveDescriptionFile(self.currentProjectName+".tdcs.json")
        else :
            saved = self.showSaveAsDialog()
        return saved

    def showSaveAsDialog(self) :
        logging.getLogger(__name__).debug("Showing save as dialog for saving description file...")
        saveFile = QFileDialog.getSaveFileName(self, "Save As", self.currentProjectName, "TDCS files (*.tdcs.json)")
        if saveFile[0] :
            saved = self.saveDescriptionFile(saveFile[0])
            if saved :
                self.currentProjectName = saveFile[0].replace(".tdcs.json", "")
                self.currentProjectSaved = True
                self.setWindowTitle(self.currentProjectName+".tdcs[*]")
        else :
            logging.getLogger(__name__).debug("Save as dialog canceled...")
            saved = False
        return saved

    def saveDescriptionFile(self, file) :
        logging.getLogger(__name__).debug("Saving description file into {}...".format(file))
        QGuiApplication.setOverrideCursor(Qt.WaitCursor)
        savedTargets = self.targets.tree.saveTargets()
        savedScene = self.central.builder.scene.saveScene()
        savedPrefSettings = self.central.scheduler.saveSettings()
        savedDescription = {**savedPrefSettings, **savedTargets, **savedScene}
        
        try :
            with open(file, "w", encoding = "utf-8") as outFile :
                json.dump(savedDescription, outFile, indent = 4, ensure_ascii = False)
        except :
            logging.getLogger(__name__).critical("While saving description file, UNEXPECTED exception occurred!\n\n", exc_info = True)
            QGuiApplication.restoreOverrideCursor()
            return False        
        
        self.setWindowModified(False)
        logging.getLogger(__name__).info("Project saved to {}.".format(file.replace(".json", "")))
        QGuiApplication.restoreOverrideCursor()
        return True

    def showNewTargetDialog(self) :
        logging.getLogger(__name__).debug("New target action triggered...")
        dialog = newTargetDialog(self)
        if dialog.exec() :
            self.targets.addNewTargets(dialog.data)
        else :
            logging.getLogger(__name__).debug("New target dialog canceled...")
            
    def toggleDragMode(self, state) :
        logging.getLogger(__name__).debug("Mode tool action triggered...")
        if self.allActions.panToolAction.isChecked() :
            self.central.builder.setDragMode(QGraphicsView.ScrollHandDrag)
        else :
            self.central.builder.setDragMode(QGraphicsView.RubberBandDrag)

    def removeSelectedItems(self) :
        logging.getLogger(__name__).debug("Remove selected action triggered...")
        selectedItems = self.central.builder.scene.selectedItems()
        logging.getLogger(__name__).debug("Removing tasks from builder scene among selected items: {}".format(selectedItems))
        for item in selectedItems :
            if item.type() == QGraphicsItem.UserType :
                self.central.builder.scene.removeTask(item)
        
        leftSelectedItems = self.central.builder.scene.selectedItems()
        if leftSelectedItems :
            logging.getLogger(__name__).debug("Removing connections from builder scene among left selected items: {}".format(leftSelectedItems))
            for item in leftSelectedItems :
                if item.type() == QGraphicsItem.UserType + 1 :
                    self.central.builder.scene.removeConnection(item)

    def onItemHelp(self) :
        QWhatsThis.enterWhatsThisMode()

    def helpContents(self) :
        logging.getLogger(__name__).debug("Help contents action triggered...")
        dialog = helpContentsDialog(self)
        dialog.show()

    def about(self) :
        logging.getLogger(__name__).debug("About action triggered...")
        dialog = aboutDialog(self)
        dialog.show()

    def zoomIn(self) :
        logging.getLogger(__name__).debug("Zoom in action triggered...")
        if self.central.currentIndex() == 0 :
            self.central.builder.zoomIn()
        else :
            self.central.scheduler.schedule.zoomIn()

    def zoomOut(self) :
        logging.getLogger(__name__).debug("Zoom out action triggered...")
        if self.central.currentIndex() == 0 :
            self.central.builder.zoomOut()
        else :
            self.central.scheduler.schedule.zoomOut()

    def resetZoom(self) :
        logging.getLogger(__name__).debug("Reset zoom action triggered...")
        if self.central.currentIndex() == 0 :
            self.central.builder.resetZoom()
        else :
            self.central.scheduler.schedule.resetZoom()

    def centralTabChanged(self, index) :
        if index == 0 :
            self.toolBar_.updateZoomButt(int(self.central.builder.zoom))
        else :
            self.toolBar_.updateZoomButt(int(self.central.scheduler.schedule.zoom))

    def sceneSelectionChanged(self) :
        logging.getLogger(__name__).debug("Builder scene selection changed...")
        self.selectedItems = self.central.builder.scene.selectedItems()
        if self.selectedItems :
            self.allActions.removeSelectedAction.setEnabled(True)
        else :
            self.allActions.removeSelectedAction.setEnabled(False)

    def builtChanged(self) :
        logging.getLogger(__name__).debug("Built changed called... ")
        if self.central.builder.scene.tasks == [] :
            self.built = True
            self.allActions.generateAction.setEnabled(False)
            return
        if self.built :
            self.built = False
            self.allActions.generateAction.setEnabled(True)

    def changeEvent(self, event) :
        if event.type() == QEvent.ModifiedChange :
            logging.getLogger(__name__).debug("ModifiedChange event invoked...")
            if self.isWindowModified() :
                self.allActions.saveAction.setEnabled(True)
            else :
                self.allActions.saveAction.setEnabled(False)
            if not self.allActions.newAction.isEnabled() :
                self.allActions.newAction.setEnabled(True)
        QMainWindow.changeEvent(self, event)

    def saveSettings(self) :
        logging.getLogger(__name__).debug("Writing settings into windows registry...")
        settings = QSettings()
        if self.isMaximized() :
            settings.setValue("maximized", 1)
        else :
            settings.setValue("maximized", 0)
            settings.setValue("geometry", self.geometry())
            settings.setValue("frame", self.frameGeometry())

    def closeEvent(self, event) :
        logging.getLogger(__name__).debug("Close action triggered...")
        if not self.checkForSave() :
                event.ignore()
                return
        self.saveSettings()
        QMainWindow.closeEvent(self, event)

    # def keyPressEvent(self, event) :
        # if event.key() == Qt.Key_Escape :
            # self.close()
            # print("boo")
            # self.variables.show()

#----------------------------------------------------------------------------#