#######################################################
##      Filename:     task_parameters_dialog.py      ##
##      Date:         07/05/2020                     ##
##      Author:       Ayman HATOUM                   ##
#######################################################
"""This file contains the class definition of taskParametersDialog(). 
A subclass of QDialog() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtWidgets import QDialog
from PyQt5.QtWidgets import QLabel
from PyQt5.QtWidgets import QLineEdit
from PyQt5.QtWidgets import QSpinBox
from PyQt5.QtWidgets import QComboBox
from PyQt5.QtWidgets import QGridLayout
from PyQt5.QtWidgets import QPushButton
from PyQt5.QtWidgets import QLayout
from PyQt5.QtWidgets import QGroupBox
from PyQt5.QtWidgets import QDoubleSpinBox
from src.constants import minInt
from src.constants import maxInt
from src.constants import minFloat
from src.constants import maxFloat
from src.constants import whatsThisMsg

#----------------------------------------------------------------------------#

class taskParametersDialog(QDialog) :
    "Task parameters dialog class"
    def __init__(self, parent, task) :
        logging.getLogger(__name__).debug("Constructing task parameters dialog...")
        super().__init__(parent)
        
        self.task = task
        self.data = {}
        
        self.setWindowTitle("Task Parameters")
        
        self.label = QLabel("Task label:", self)
        if task.label :
            self.labelInput = QLineEdit(task.label.toPlainText(), self)
            self.labelInput.selectAll()
        else :
            self.labelInput = QLineEdit(self)
        self.labelInput.setPlaceholderText("Type a label...")
        self.label.setWhatsThis(whatsThisMsg.format("Task Label", "Used for user differentiation, it can be <strong>non-unique</strong> and even an <strong>empty</strong> string."))
        self.labelInput.setWhatsThis(whatsThisMsg.format("Task Label", "Used for user differentiation, it can be <strong>non-unique</strong> and even an <strong>empty</strong> string."))
        
        self.okButt = QPushButton("OK", self)
        self.okButt.setDefault(True)
        self.okButt.clicked.connect(self.accept)
        self.cancelButt = QPushButton("Cancel", self)
        self.cancelButt.clicked.connect(self.reject)

        self.layout = QGridLayout(self)
        self.layout.addWidget(self.label, 0, 0)
        self.layout.addWidget(self.labelInput, 0, 1, 1, 4)

        if task.inputs :
            self.inputsDefaultValues = QGroupBox("Inputs Default Values", self)
            self.inputsDefaultValues.setWhatsThis(whatsThisMsg.format("Inputs Default Values", "Assign inputs default values, depending on their type."))
            self.inputsNames = []
            self.inputsValues = []
            self.boxLayout = QGridLayout()
            for port, row in zip(task.inputs, range(len(task.inputs))) :
                name = QLabel(task.name+"_input_"+port.label+" ("+port.typ+"):", self)
                self.inputsNames.append(name)
                self.boxLayout.addWidget(name, row, 0)
                if port.typ == "int" :
                    value = QSpinBox(self)
                    if port.minValue :
                        MIN = port.minValue
                    else :
                        MIN = minInt
                    value.setMinimum(MIN)
                    if port.maxValue :
                        MAX = port.maxValue
                    else :
                        MAX = maxInt
                    value.setMaximum(MAX)
                    value.setToolTip("<b>MIN:</b> "+str(MIN)+"<br><b>MAX:</b> "+str(MAX))
                    value.setValue(port.defaultValue)
                elif port.typ == "float" :
                    value = QDoubleSpinBox(self)
                    if port.minValue :
                        MIN = port.minValue
                    else :
                        MIN = minFloat
                    value.setMinimum(MIN)
                    if port.maxValue :
                        MAX = port.maxValue
                    else :
                        MAX = maxFloat
                    value.setMaximum(MAX)
                    value.setToolTip("<b>MIN:</b> "+str(MIN)+"<br><b>MAX:</b> "+str(MAX))
                    value.setValue(port.defaultValue)
                else :
                    value = QComboBox(self)
                    value.addItems(["True", "False"])
                    if port.defaultValue :
                        value.setCurrentIndex(0)
                    else :
                        value.setCurrentIndex(1)
                self.inputsValues.append(value)
                self.boxLayout.addWidget(value, row, 1)
            self.inputsDefaultValues.setLayout(self.boxLayout)                    
        
            self.layout.addWidget(self.inputsDefaultValues, 1, 0, 1, 5)
            self.layout.addWidget(self.okButt, 2, 3)
            self.layout.addWidget(self.cancelButt, 2, 4)           
        else :        
            self.layout.addWidget(self.okButt, 1, 3)
            self.layout.addWidget(self.cancelButt, 1, 4)
        
        self.layout.setSpacing(10)
        self.layout.setColumnMinimumWidth(3, 100)
        self.layout.setColumnMinimumWidth(4, 100)
        self.layout.setSizeConstraint(QLayout.SetFixedSize)
        
        self.setLayout(self.layout)
        
    def accept(self) :        
        self.data[self.label.text().strip(":")] = self.labelInput.text()
        if self.task.inputs :
            for port, value in zip(self.task.inputs, self.inputsValues) :
                if port.typ == "bool" :
                    self.data[port.label] = True if value.currentText() == "True" else False
                else :
                    self.data[port.label] = value.value()
                    
        QDialog.accept(self)
                
#----------------------------------------------------------------------------#