###########################################################
##      Filename:     preference_settings_dialog.py      ##
##      Date:         19/06/2020                         ##
##      Author:       Ayman HATOUM                       ##
###########################################################
"""This file contains the class definition of preferenceSettingsDialog(). 
A subclass of QDialog() from PyQt5."""

__author__ = "KAI"

#----------------------------------------------------------------------------#

###############
##  Imports  ##
###############
import logging
from PyQt5.QtWidgets import QDialog
from PyQt5.QtWidgets import QLabel
from PyQt5.QtWidgets import QSpinBox
from PyQt5.QtWidgets import QComboBox
from PyQt5.QtWidgets import QGridLayout
from PyQt5.QtWidgets import QPushButton
from PyQt5.QtWidgets import qApp
from PyQt5.QtWidgets import QLayout
from PyQt5.QtWidgets import QGroupBox
from PyQt5.QtWidgets import QHBoxLayout
from PyQt5.QtWidgets import QVBoxLayout
from src.constants import canBaudRates
from src.constants import defaultBaudRate
from src.constants import timeUnits
from src.constants import defaultTimeUnit
from src.constants import defaultSlotBuff
from src.constants import defaultRoundBuff
from src.constants import slotRoutines
from src.constants import deafaultSlotRoutine
from src.constants import defaultSlotOverlapBound
from src.constants import whatsThisMsg

#----------------------------------------------------------------------------#

class preferenceSettingsDialog(QDialog) :
    "Preference settings dialog class"
    def __init__(self, parent) :
        logging.getLogger(__name__).debug("Constructing preference settings dialog...")
        super().__init__(parent)
        
        self.mainWindow = parent
        self.data = {}
        
        self.setWindowTitle("Preference Settings")
        
        self.schedulerParam = QGroupBox("Scheduler Parameters")
        self.schedulerParam.setWhatsThis(whatsThisMsg.format("Scheduler Parameters", "Used for generating schedule."))
        
        self.baudRate = QLabel("Baud rate:", self)
        self.baudRateInput = QComboBox(self)
        self.baudRateInput.addItems(canBaudRates)
        self.baudRateInput.setCurrentText(self.mainWindow.central.scheduler.baudRate)
        self.baudRate.setWhatsThis(whatsThisMsg.format("Baud Rate", "You can choose one of the <strong>CAN Bus</strong> available baud rates."))
        self.baudRateInput.setWhatsThis(whatsThisMsg.format("Baud Rate", "You can choose one of the <strong>CAN Bus</strong> available baud rates."))

        self.timeUnit = QLabel("Time unit:", self)
        self.timeUnitInput = QComboBox(self)
        self.timeUnitInput.addItems(timeUnits)
        self.timeUnitInput.setCurrentText(self.mainWindow.central.scheduler.timeUnit)
        self.timeUnit.setWhatsThis(whatsThisMsg.format("Time Unit", "Defines the time unit for reading config files of <strong>targets</strong>. As well as for schedule's table data."))
        self.timeUnitInput.setWhatsThis(whatsThisMsg.format("Time Unit", "Defines the time unit for reading config files of <strong>targets</strong>. As well as for schedule's table data."))

        self.slotRoutine = QLabel("Slot routine:", self)
        self.slotRoutineInput = QComboBox(self)
        self.slotRoutineInput.addItems(slotRoutines)
        self.slotRoutineInput.setCurrentText(self.mainWindow.central.scheduler.slotRoutine)
        self.slotRoutine.setWhatsThis(whatsThisMsg.format("Slot Routine", "You can choose to have either <strong>Variable</strong> length slots or <strong>Fixed</strong> length slots."))
        self.slotRoutineInput.setWhatsThis(whatsThisMsg.format("Slot Routine", "You can choose to have either <strong>Variable</strong> length slots or <strong>Fixed</strong> length slots."))

        self.slotBuff = QLabel("Slot buffer:", self)
        self.slotBuffInput = QSpinBox(self)
        self.slotBuffInput.setMaximum(100000)
        self.slotBuffInput.setValue(self.mainWindow.central.scheduler.slotBuff)
        self.slotBuff.setWhatsThis(whatsThisMsg.format("Slot Buffer", "Defines <strong>same</strong> buffer for all slots."))
        self.slotBuffInput.setWhatsThis(whatsThisMsg.format("Slot Buffer", "Defines <strong>same</strong> buffer for all slots."))

        self.schedulerParamBox = QGridLayout()
        self.schedulerParamBox.addWidget(self.baudRate, 0, 0)
        self.schedulerParamBox.addWidget(self.baudRateInput, 0, 1)
        self.schedulerParamBox.addWidget(self.timeUnit, 1, 0)
        self.schedulerParamBox.addWidget(self.timeUnitInput, 1, 1)
        self.schedulerParamBox.addWidget(self.slotRoutine, 2, 0)
        self.schedulerParamBox.addWidget(self.slotRoutineInput, 2, 1)
        self.schedulerParamBox.addWidget(self.slotBuff, 3, 0)
        self.schedulerParamBox.addWidget(self.slotBuffInput, 3, 1)
        self.schedulerParamBox.setSpacing(10)
        self.schedulerParam.setLayout(self.schedulerParamBox)
        
        self.scheduleScriptParam = QGroupBox("Schedule Script Parameters")
        self.scheduleScriptParam.setWhatsThis(whatsThisMsg.format("Schedule Script Parameters", "Used for generating the schedule script."))

        self.slotOverlapBound = QLabel("Slot overlap bound:", self)
        self.slotOverlapBoundInput = QSpinBox(self)
        self.slotOverlapBoundInput.setMaximum(100000)
        self.slotOverlapBoundInput.setValue(self.mainWindow.central.scheduler.slotOverlapBound)
        self.slotOverlapBound.setWhatsThis(whatsThisMsg.format("Slot Overlap Bound", "Defines <strong>same</strong> overlap bound for all slots. This parameter <strong>doesn't</strong> affect schedule generation."))
        self.slotOverlapBoundInput.setWhatsThis(whatsThisMsg.format("Slot Overlap Bound", "Defines <strong>same</strong> overlap bound for all slots. This parameter <strong>doesn't</strong> affect schedule generation."))

        self.roundBuff = QLabel("Round buffer:", self)
        self.roundBuffInput = QSpinBox(self)
        self.roundBuffInput.setMaximum(100000)
        self.roundBuffInput.setValue(self.mainWindow.central.scheduler.roundBuff)
        self.roundBuff.setWhatsThis(whatsThisMsg.format("Round Buffer", "Defines buffer for the <strong>complete round</strong> <i>(hyperperiod)</i>. This parameter <strong>doesn't</strong> affect schedule generation."))
        self.roundBuffInput.setWhatsThis(whatsThisMsg.format("Round Buffer", "Defines buffer for the <strong>complete round</strong> <i>(hyperperiod)</i>. This parameter <strong>doesn't</strong> affect schedule generation."))
        
        self.scheduleScriptParamBox = QGridLayout()
        self.scheduleScriptParamBox.addWidget(self.slotOverlapBound, 0, 0)
        self.scheduleScriptParamBox.addWidget(self.slotOverlapBoundInput, 0, 1)
        self.scheduleScriptParamBox.addWidget(self.roundBuff, 1, 0)
        self.scheduleScriptParamBox.addWidget(self.roundBuffInput, 1, 1)
        self.scheduleScriptParamBox.setSpacing(10)
        self.scheduleScriptParam.setLayout(self.scheduleScriptParamBox)

        self.okButt = QPushButton("OK", self)
        self.okButt.setDefault(True)
        self.okButt.clicked.connect(self.accept)
        self.resetButt = QPushButton("Reset", self)
        self.resetButt.clicked.connect(self.reset)
        self.cancelButt = QPushButton("Cancel", self)
        self.cancelButt.clicked.connect(self.reject)

        self.buttonsLayout = QHBoxLayout()
        self.buttonsLayout.addWidget(self.okButt)
        self.buttonsLayout.addWidget(self.resetButt)
        self.buttonsLayout.addWidget(self.cancelButt)
        self.buttonsLayout.setSpacing(10)

        self.layout = QVBoxLayout()
        self.layout.addWidget(self.schedulerParam)
        self.layout.addWidget(self.scheduleScriptParam)
        self.layout.addLayout(self.buttonsLayout)
        self.layout.setSpacing(10)
        self.layout.setSizeConstraint(QLayout.SetFixedSize)
        self.setLayout(self.layout)
        
    def accept(self) :
        self.data[self.baudRate.text().strip(":")] = self.baudRateInput.currentText()
        self.data[self.timeUnit.text().strip(":")] = self.timeUnitInput.currentText()
        self.data[self.slotRoutine.text().strip(":")] = self.slotRoutineInput.currentText()
        self.data[self.slotBuff.text().strip(":")] = self.slotBuffInput.value()
        self.data[self.slotOverlapBound.text().strip(":")] = self.slotOverlapBoundInput.value()
        self.data[self.roundBuff.text().strip(":")] = self.roundBuffInput.value()
        QDialog.accept(self)

    def reset(self) :
        logging.getLogger(__name__).debug("Reset preference settings...")
        self.baudRateInput.setCurrentText(defaultBaudRate)
        self.timeUnitInput.setCurrentText(defaultTimeUnit)
        self.slotRoutineInput.setCurrentText(deafaultSlotRoutine)
        self.slotBuffInput.setValue(defaultSlotBuff)
        self.slotOverlapBoundInput.setValue(defaultSlotOverlapBound)
        self.roundBuffInput.setValue(defaultRoundBuff)
                
#----------------------------------------------------------------------------#