# Some of the functions used were taken from: https://github.com/carla-simulator/carla
# The implemented functions have been modified and improved in order to perform some necessary tasks for the purpose of the thesis

"""
Welcome to CARLA ADAS_scenario.
Keyboard controls:

    P            : toggle autopilot
    TAB          : change sensor position
    H            : toggle help
    ESC/Q        : quit
    F1           : toggle HUD
    C            : change weather
    R            : toggle recording images
    N            : next sensor
    I            : take a snapshot of the scene
    
"""

from __future__ import print_function

# ==============================================================================
# -- find carla module ---------------------------------------------------------
# ==============================================================================


import glob
import os
import sys

try:
    sys.path.append(glob.glob('../carla/dist/carla-*%d.%d-%s.egg' % (
        sys.version_info.major,
        sys.version_info.minor,
        'win-amd64' if os.name == 'nt' else 'linux-x86_64'))[0])
except IndexError:
    pass


# ==============================================================================
# -- imports -------------------------------------------------------------------
# ==============================================================================


import carla

from carla import ColorConverter as cc

import time
import argparse
import collections
import datetime
import logging
import math
import random
import re
import weakref
import math
import cv2

# Keyboard commands
try:
    import pygame
    from pygame.locals import KMOD_SHIFT
    from pygame.locals import K_ESCAPE
    from pygame.locals import K_F1
    from pygame.locals import K_TAB
    from pygame.locals import K_c
    from pygame.locals import K_n
    from pygame.locals import K_h
    from pygame.locals import K_p
    from pygame.locals import K_q
    from pygame.locals import K_r
    from pygame.locals import K_i
    
except ImportError:
    raise RuntimeError('cannot import pygame, make sure pygame package is installed')

try:
    import numpy as np
except ImportError:
    raise RuntimeError('cannot import numpy, make sure numpy package is installed')

# ==============================================================================
# -- Global variables ----------------------------------------------------------
# ==============================================================================

HOST = 'localhost'
PORT = 2000
TIMEOUT = 10.0  # seconds
num = 0         # file number
j = 0
all_actors=[]

# ==============================================================================
# -- Global functions ----------------------------------------------------------
# ==============================================================================

def find_weather_presets():
    """
    ## This function is used to set the initial weather and change it when the C key is pressed
    """
    rgx = re.compile('.+?(?:(?<=[a-z])(?=[A-Z])|(?<=[A-Z])(?=[A-Z][a-z])|$)')
    name = lambda x: ' '.join(m.group(0) for m in rgx.finditer(x))
    presets = [x for x in dir(carla.WeatherParameters) if re.match('[A-Z].+', x)]
    return [(getattr(carla.WeatherParameters, x), name(x)) for x in presets]


def get_actor_display_name(actor, truncate=250):
    """
    ## This function is used to search for the name of the ego-vehicle
    """
    name = ' '.join(actor.type_id.replace('_', '.').title().split('.')[1:])
    return (name[:truncate - 1] + u'\u2026') if len(name) > truncate else name


# ==============================================================================
# -- World ---------------------------------------------------------------------
# ==============================================================================

class World(object):
    """
    ## This class is the main class, responsible for the creation of all the actors (vehicles, pedestrians and sensors)
    ## present in the simulation
    """
    def __init__(self, carla_world, hud, args, vehicles_list, walkers_list, all_id, client):
        self.world = carla_world
        all_actors = self.world.get_actors()
        try:
            self.map = self.world.get_map()
        except RuntimeError as error:
            print('RuntimeError: {}'.format(error))
            print('  The server could not send the OpenDRIVE (.xodr) file:')
            print('  Make sure it exists, has the same name of your town, and is correct.')
            sys.exit(1)
        self.hud = hud
        self.number_of_vehicles = args.number_of_vehicles
        self.number_of_pedestrians = args.number_of_walkers
        self.vehicles_list = vehicles_list
        self.walkers_list = walkers_list
        self.all_id = all_id
        self.client = client
        self.ego_vehicle = None
        self.camera_manager = None
        self._weather_presets = find_weather_presets()
        self._weather_index = 0
        self._gamma = args.gamma
        self.restart() # start
        self.world.on_tick(hud.on_world_tick)
        
    # START
    def restart(self):
        """
        ## This function is used to start the simulation creating and positioning the actors with specified features
        """
        global all_actors
        # Keep same camera configuration if the camera manager exists
        cam_index = self.camera_manager.index if self.camera_manager is not None else 0
        cam_pos_index = self.camera_manager.transform_index if self.camera_manager is not None else 0
        
        # Get the number of traffic speed limits
        act_list = self.world.get_actors()
        speed_limit = act_list.filter('traffic.speed_limit.*')
        print('Number of speed limit in the actual world:', len(speed_limit))
    
        # Set the initial weather conditions
        weather = carla.WeatherParameters(cloudyness=10.0,
                                          precipitation=0.0,
                                          precipitation_deposits=0.0,
                                          wind_intensity=10.0,
                                          sun_azimuth_angle=180.0,
                                          sun_altitude_angle=70.0)
        self.world.set_weather(weather)
        print('Initial Weather: %s' % self.world.get_weather())
        
        # Destroy all the previous actors before starting the simulation
        actors = self.world.get_actors().filter('vehicle.*.*')
        for actor in actors:
            actor.destroy()
        
        # Get a correct blueprint
        blueprint_library = self.world.get_blueprint_library()
        
        # Creating the ego-vehicle setting its color and its position
        blueprint = blueprint_library.find('vehicle.tesla.model3')
        # Changes attributes
        if blueprint.has_attribute('color'):
            cornflowerblue_color = ('100,149,237')
            blueprint.set_attribute('color', cornflowerblue_color)
        if blueprint.has_attribute('driver_id'):
            driver_id = random.choice(blueprint.get_attribute('driver_id').recommended_values)
            blueprint.set_attribute('driver_id', driver_id)
        if blueprint.has_attribute('is_invincible'):
            blueprint.set_attribute('is_invincible', 'true')
            
        # Spawn the ego_vehicle at the desired location and the spectator camera at the actor's location
        if self.ego_vehicle is not None:
            spawn_point = self.ego_vehicle.get_transform()
            spawn_point.location.z += 2.0
            spawn_point.rotation.roll = 0.0
            spawn_point.rotation.pitch = 0.0
            self.destroy()
            # Spawn the vehicle at the desired position
            transform = carla.Transform(carla.Location(x=244.0, y=-169.0, z=2.0))
            self.ego_vehicle = self.world.spawn_actor(blueprint, transform)
            self.ego_vehicle.set_simulate_physics(True)
            
        while self.ego_vehicle is None:
            if not self.map.get_spawn_points():
                print('There are no spawn points available in your map/town.')
                print('Please add some Vehicle Spawn Point to your UE4 scene.')
                sys.exit(1)
            spawn_points = self.map.get_spawn_points()
            # Spawn the vehicle at the desired position
            spawn_point = carla.Transform(carla.Location(x=244.0, y=-169.0, z=2.0))
            transform = spawn_point
            self.ego_vehicle = self.world.spawn_actor(blueprint, spawn_point)
            self.ego_vehicle.set_simulate_physics(True)
            
        # Get the spectator
        spectator = self.world.get_spectator()
        # Set spectator position at given transform (vehicle transform)
        transform.location.x -= 14
        transform.location.y -= 3
        transform.location.z += 15
        transform.rotation.pitch = -30
        spectator.set_transform(transform)
        
        # Wait for world to get the ego-vehicle actor
        self.world.tick()
        self.world.wait_for_tick()
        
        # Get the location and the center of mass of ego-vehicle
        loc = self.ego_vehicle.get_location()
        print('\nLocation: ', loc)
        print ("%s %s %s created and positioned" %(blueprint.tags[0], 
												   blueprint.tags[1], 
												   blueprint.tags[2]))
        
        print("Ego vehicle center of mass: ", self.ego_vehicle.bounding_box.location)
        print("Ego vehicle extention: ", self.ego_vehicle.bounding_box.extent)
        
        # Sets the physical characteristics of the ego-vehicle (tesla model 3 -> features available at: https://www.tesla.com/it_it/model3)
        # Create front and rear Wheels Physics Control for the ego-vehicle
        front_left_wheel  = carla.WheelPhysicsControl(tire_friction=4.5, damping_rate=1.0)
        front_right_wheel = carla.WheelPhysicsControl(tire_friction=4.5, damping_rate=1.0)
        rear_left_wheel   = carla.WheelPhysicsControl(tire_friction=4.5, damping_rate=1.0, max_steer_angle=0.0)
        rear_right_wheel  = carla.WheelPhysicsControl(tire_friction=4.5, damping_rate=1.0, max_steer_angle=0.0)

        wheels = [front_left_wheel, front_right_wheel, rear_left_wheel, rear_right_wheel]
		
        # Change Vehicle Physics Control parameters of the ego-vehicle
        physics_control = self.ego_vehicle.get_physics_control()
        physics_control.max_rpm = 10000.0
        physics_control.mass = 1847.0
        physics_control.use_gear_autobox = True
        physics_control.drag_coefficient = 0.23
        physics_control.wheels = wheels

        # Apply Vehicle Physics Control for the ego-vehicle
        self.ego_vehicle.apply_physics_control(physics_control)
        
        # Prepare to spawn other vehicles and pedestrians
        blueprintsVehicles = self.world.get_blueprint_library().filter('vehicle.*')
        blueprintsWalkers = self.world.get_blueprint_library().filter('walker.pedestrian.*')
        spawn_points = self.map.get_spawn_points()
        number_of_spawn_points = len(spawn_points)
        
        # Check the number of spawn points
        if self.number_of_vehicles < number_of_spawn_points:
            random.shuffle(spawn_points)
        elif self.number_of_vehicles > number_of_spawn_points:
            msg = 'requested %d vehicles, but could only find %d spawn points'
            logging.warning(msg, self.number_of_vehicles, number_of_spawn_points)
            self.number_of_vehicles = number_of_spawn_points

        # Set the commands
        SpawnActor = carla.command.SpawnActor
        SetAutopilot = carla.command.SetAutopilot
        FutureActor = carla.command.FutureActor
        
        # --------------------
        # Spawn other vehicles
        # --------------------
        
        batch = []
        for n, transform in enumerate(spawn_points):
            if n >= self.number_of_vehicles:
                break
            # Take random vehicles
            blueprint = random.choice(blueprintsVehicles)
            if blueprint.has_attribute('color'):
                color = random.choice(blueprint.get_attribute('color').recommended_values)
                blueprint.set_attribute('color', color)
            if blueprint.has_attribute('driver_id'):
                driver_id = random.choice(blueprint.get_attribute('driver_id').recommended_values)
                blueprint.set_attribute('driver_id', driver_id)
            blueprint.set_attribute('role_name', 'autopilot')
            batch.append(SpawnActor(blueprint, transform).then(SetAutopilot(FutureActor, True)))

        for response in self.client.apply_batch_sync(batch):
            if response.error:
                logging.error(response.error)
            else:
                self.vehicles_list.append(response.actor_id)
        
        # ------------------
        # Spawn Pedestrians
        # ------------------
        
        # 1. take all the random locations to spawn
        spawn_points = []
        for i in range(self.number_of_pedestrians):
            spawn_point = carla.Transform()
            loc = self.world.get_random_location_from_navigation()
            if (loc != None):
                spawn_point.location = loc
                spawn_points.append(spawn_point)
        # 2. we spawn the walker object
        batch = []
        for spawn_point in spawn_points:
            walker_bp = random.choice(blueprintsWalkers)
            # set walker attribute as not invencible
            if walker_bp.has_attribute('is_invincible'):
                walker_bp.set_attribute('is_invincible', 'false')
            batch.append(SpawnActor(walker_bp, spawn_point))
        results = self.client.apply_batch_sync(batch, True)
        for i in range(len(results)):
            if results[i].error:
                logging.error(results[i].error)
            else:
                self.walkers_list.append({"id": results[i].actor_id})
        # 3. we spawn the walker controller
        batch = []
        walker_controller_bp = self.world.get_blueprint_library().find('controller.ai.walker')
        for i in range(len(self.walkers_list)):
            batch.append(SpawnActor(walker_controller_bp, carla.Transform(), self.walkers_list[i]["id"]))
        results = self.client.apply_batch_sync(batch, True)
        for i in range(len(results)):
            if results[i].error:
                logging.error(results[i].error)
            else:
                self.walkers_list[i]["con"] = results[i].actor_id
        # 4. we put altogether the walkers and controllers id to get the objects from their id
        for i in range(len(self.walkers_list)):
            self.all_id.append(self.walkers_list[i]["con"])
            self.all_id.append(self.walkers_list[i]["id"])
        all_actors = self.world.get_actors(self.all_id)

        # wait for a tick to ensure client receives the last transform of the walkers we have just created
        self.world.wait_for_tick()
        # data updated
        # 5. initialize each controller and set target to walk to (list is [controler, actor, controller, actor ...])
        for i in range(0, len(self.all_id), 2):
            # start walker
            all_actors[i].start()
            # set walk to random point
            all_actors[i].go_to_location(self.world.get_random_location_from_navigation())
            # random max speed
            all_actors[i].set_max_speed(1 + random.random()/2)    # max speed between 1 and 1.5 (default is 1.4 m/s)
        
        print('\nSpawned %d vehicles and %d pedestrians' % (len(self.vehicles_list), len(self.walkers_list)))
        
        # Update the world
        self.world.tick()
        self.world.wait_for_tick()
        
        # Set up the sensors and the camera index
        self.camera_manager = CameraManager(self.ego_vehicle, self.hud, self._gamma)
        self.camera_manager.transform_index = cam_pos_index
        self.camera_manager.set_sensor(cam_index, notify=False)
        
        # Display the ego-vehicle's name
        actor_type = get_actor_display_name(self.ego_vehicle)
        self.hud.notification(actor_type + "       For displaying more information Press 'H' for help.")

    def next_weather(self, reverse=False):
        """
        ## Funtion used to change the weather of the simulation when C key is pressed
        """
        self._weather_index += -1 if reverse else 1
        self._weather_index %= len(self._weather_presets)
        preset = self._weather_presets[self._weather_index]
        self.hud.notification('Weather: %s' % preset[1])
        self.ego_vehicle.get_world().set_weather(preset[0])

    def tick(self, clock):
        """
        # Function used to update the python HUD
        """
        self.hud.tick(self, clock)

    def render(self, display):
        """
        ## Function used to show HUD information on the python window screen
        """
        self.camera_manager.render(display)
        self.hud.render(display)

    def destroy_sensors(self):
        """
        ## Function used to destroy all the sensors at the end of simulation
        """
        self.camera_manager.sensor.destroy()
        self.camera_manager.sensor = None
        self.camera_manager.index = None

    def destroy(self):
        """
        ## Function used to destroy all the actors at the end of simulation
        """
        actors = [
            self.camera_manager.sensor,
            self.ego_vehicle]
        for actor in actors:
            if actor is not None:
                actor.destroy()

# ==============================================================================
# -- KeyboardControl -----------------------------------------------------------
# ==============================================================================

class KeyboardControl(object):
    """
    ## This class is used to control and parse the keyboard command event
    """
    def __init__(self, world, start_in_autopilot):
        self._autopilot_enabled = start_in_autopilot
        if isinstance(world.ego_vehicle, carla.Vehicle):
            self._control = carla.VehicleControl()
            world.ego_vehicle.set_autopilot(self._autopilot_enabled)
        else:
            raise NotImplementedError("Actor type not supported")
        self.tag = 0

    def parse_events(self, client, world, clock):
        """
        ## This function is used to get the keys pushed by the user activating a particular function
        """
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                return True
            elif event.type == pygame.KEYUP:
                if self._is_quit_shortcut(event.key):
                    return True
                # activate/deactivate information about the simulation on screen
                elif event.key == K_F1:
                    world.hud.toggle_info()
                # activate/deactivate helper on screen
                elif event.key == K_h:
                    world.hud.help.toggle()
                # switch views
                elif event.key == K_TAB:
                    world.camera_manager.toggle_camera()
                    if self.tag < 1:
                        world.hud.notification('Sensor Camera front view', seconds=1.5)
                        self.tag = self.tag + 1
                    else:
                        self.tag = 0
                # change the weather forwards or backwards (by pressing SHIFT+C command)
                elif event.key == K_c and pygame.key.get_mods() & KMOD_SHIFT:
                    world.next_weather(reverse=True)
                elif event.key == K_c:
                    world.next_weather()
                # activate/deactivate snapshot of the scene
                elif event.key == K_i:
                    world.camera_manager.toggle_snapshot()
                # activate/deactivate images recording
                elif event.key == K_r:
                    world.camera_manager.toggle_recording()
                # change sensors
                elif event.key == K_n:
                    world.camera_manager.next_sensor()
                # activate/deactivate autopilot mode (applied to the ego-vehicle)
                if isinstance(self._control, carla.VehicleControl):
                    if event.key == K_p:
                        self._autopilot_enabled = not self._autopilot_enabled
                        world.ego_vehicle.set_autopilot(self._autopilot_enabled)
                        world.hud.notification('Autopilot %s' % ('On' if self._autopilot_enabled else 'Off'))

    @staticmethod
    def _is_quit_shortcut(key):
        # Quit the simulation
        return (key == K_ESCAPE) or (key == K_q)

# ==============================================================================
# -- HUD -----------------------------------------------------------------------
# ==============================================================================

class HUD(object):
    """
    ## This class is used to set the HUD on the left part of the pygame window screen getting and displaying the useful information about the simulation
    """
    def __init__(self, width, height):
        self.dim = (width, height)
        # Set the font of the text
        get_font = pygame.font.get_default_font()
        # all available fonts: print(pygame.font.get_fonts())
        font = pygame.font.Font(pygame.font.get_default_font(), 20)
        # print('Font: ', pygame.font.get_default_font())
        # font = pygame.font.Font('freesansbold.ttf', 20)
        fonts = [x for x in pygame.font.get_fonts() if 'mono' in x]
        default_font = 'ubuntumono'
        mono = default_font if default_font in fonts else fonts[0]
        mono = pygame.font.match_font(mono)
        self._font_mono = pygame.font.Font(mono, 14)
        self._notifications = FadingText(font, (width, 40), (0, height - 40))
        self.help = HelpText(pygame.font.Font(mono, 24), width, height)
        self.server_fps = 0
        self.frame = 0
        self.simulation_time = 0
        self._show_info = True
        self._info_text = []
        self.affected_traffic_light_text = 'None'
        self._server_clock = pygame.time.Clock()

    def on_world_tick(self, timestamp):
        """
        ## Function used to update the server information
        """
        self._server_clock.tick()
        self.server_fps = self._server_clock.get_fps()
        self.frame = timestamp.frame
        self.simulation_time = timestamp.elapsed_seconds

    def tick(self, world, clock):
        """
        ## Function used to display some relevant information about the simulation on the HUD screen
        """
        self._notifications.tick(world, clock)
        if not self._show_info:
            return
        # Get the position and velocity of the ego-vehicle
        t = world.ego_vehicle.get_transform()
        v = world.ego_vehicle.get_velocity()
        # Get the speed limit of which the ego-vehicle is affected
        su = world.ego_vehicle.get_speed_limit()
        # Get the lane width using waypoints
        location = world.ego_vehicle.get_location()
        smap = world.world.get_map()
        vechicle_waypoint = smap.get_waypoint(location)
        lane_width = vechicle_waypoint.lane_width
        
        # Get the vehicle lateral error using waypoints
        right_lane_waypoint = vechicle_waypoint.get_right_lane()
        if right_lane_waypoint is not None:
            e1 = math.sqrt((right_lane_waypoint.transform.location.x -location.x)**2 + \
					    (right_lane_waypoint.transform.location.y - location.y)**2) - \
					    lane_width/2
        else:
            e1 = ''
        # Get the traffic light of which the ego-vehicle is affected
        tl = world.ego_vehicle.get_traffic_light()
        if tl is not None and tl.get_state() == carla.TrafficLightState.Red:
            traf = 'Red'
        elif tl is not None and tl.get_state() == carla.TrafficLightState.Yellow:
            traf = 'Yellow'
        else:
            traf='Green'
        
        vehicles = world.world.get_actors().filter('vehicle.*')
        sensors = world.world.get_actors().filter('sensor.*')
        walkers = world.world.get_actors().filter('walker.*')
        
        # Show useful text information on the HUD screen
        self._info_text = [
            'Server:  % 16.0f FPS' % self.server_fps,
            'Client:  % 16.0f FPS' % clock.get_fps(),
            '',
            'Map:     % 20s' % world.map.name,
            'Simulation time: % 12s' % datetime.timedelta(seconds=int(self.simulation_time)),
            '',
            'Vehicle: % 20s' % get_actor_display_name(world.ego_vehicle, truncate=20),
            'Vehicle Speed: % 8.0f km/h' % (3.6 * math.sqrt(v.x**2 + v.y**2 + v.z**2)),
            'Location:% 20s' % ('(% 5.1f, % 5.1f)' % (t.location.x, t.location.y)),
            'Number of vehicles: % 8d' % len(vehicles),
            'Number of walkers:  % 8d' % len(walkers),
            'Vehicle Affected by:',
            '  Speed Limit:       %3d km/h' % su,
            '  Traffic Light:     % 5s'      % traf,
            '  Lateral Error      %.5s m'   % e1,
            '  Lane Width:         %.2f m'   % lane_width,
            '']
            
        if len(vehicles) > 1:
            # Shows vehicles near the ego-vehicle and their distance
            self._info_text += ['Nearby vehicles:']
            distance = lambda l: math.sqrt((l.x - t.location.x)**2 + (l.y - t.location.y)**2 + (l.z - t.location.z)**2)
            vehicles = [(distance(x.get_location()), x) for x in vehicles if x.id != world.ego_vehicle.id]
            for d, vehicle in sorted(vehicles):
                if d > 200.0:
                    break
                vehicle_type = get_actor_display_name(vehicle, truncate=22)
                self._info_text.append('% 4dm %s' % (d, vehicle_type))

    def toggle_info(self):
        # Activate/Deactivate information display
        self._show_info = not self._show_info

    def notification(self, text, seconds=2.0):
        """
        ## Function used to set the notification text (its duration is in seconds)
        """
        self._notifications.set_text(text, seconds=seconds)

    def error(self, text):
        self._notifications.set_text('Error: %s' % text, (255, 0, 0))

    def render(self, display):
        """
        ## Function used to set the HUD display parameters
        """
        if self._show_info:
            info_surface = pygame.Surface((255, self.dim[1]))
            info_surface.set_alpha(100)
            display.blit(info_surface, (0, 0))
            v_offset = 7
            for item in self._info_text:
                if v_offset + 18 > self.dim[1]:
                    break
                if item:  # At this point has to be a str
                    surface = self._font_mono.render(item, True, (255, 215, 0))
                    display.blit(surface, (8, v_offset))
                v_offset += 22
        self._notifications.render(display)
        self.help.render(display)

# ==============================================================================
# -- FadingText ----------------------------------------------------------------
# ==============================================================================

class FadingText(object):
    """
    ## This class is used to show on the bottom of the screen the notification of events (keyboard command)
    """
    def __init__(self, font, dim, pos):
        self.font = font
        self.dim = dim
        self.pos = pos
        self.seconds_left = 0
        self.surface = pygame.Surface(self.dim)

    def set_text(self, text, color=(255, 255, 255), seconds=2.0):
        """
        ## Function used to set the text on the notification event area
        """
        text_texture = self.font.render(text, True, color)
        self.surface = pygame.Surface(self.dim)
        self.seconds_left = seconds
        self.surface.fill((0, 0, 0, 0))
        self.surface.blit(text_texture, (10, 11))

    def tick(self, _, clock):
        delta_seconds = 1e-3 * clock.get_time()
        self.seconds_left = max(0.0, self.seconds_left - delta_seconds)
        self.surface.set_alpha(500.0 * self.seconds_left)

    def render(self, display):
        """
        ## Function used to draw an image/surface on the screen
        """
        display.blit(self.surface, self.pos)

# ==============================================================================
# -- HelpText ------------------------------------------------------------------
# ==============================================================================

class HelpText(object):
    """
    ## This class is used to display the helper text on a surface in the center of the screen
    """
    def __init__(self, font, width, height):
        lines = __doc__.split('\n')
        self.font = font
        self.dim = (680, len(lines) * 22 + 12)
        self.pos = (0.5 * width - 0.5 * self.dim[0], 0.5 * height - 0.5 * self.dim[1])
        self.seconds_left = 0
        self.surface = pygame.Surface(self.dim)
        self.surface.fill((60, 60, 60, 0))
        for n, line in enumerate(lines):
            text_texture = self.font.render(line, True, (255, 255, 255))
            self.surface.blit(text_texture, (22, n * 22))
            self._render = False
        self.surface.set_alpha(220)

    def toggle(self):
        # Activate/Deactivate helper Text
        self._render = not self._render

    def render(self, display):
        # Display the helper text on the screen
        if self._render:
            display.blit(self.surface, self.pos)

# ==============================================================================
# -- CameraManager -------------------------------------------------------------
# ==============================================================================

class CameraManager(object):
    """
    ## This class is used to set the sensors with their characteristics and attributes,
    ## establishing a method to store sensor information (image) from the screen on the disk
    """
    def __init__(self, parent_actor, hud, gamma_correction):
        self.sensor = None
        self.surface = None
        self._parent = parent_actor
        self.hud = hud
        self.recording = False
        self.snapshot = False
        bound_y = 0.5 + self._parent.bounding_box.extent.y
        Attachment = carla.AttachmentType
        
        # Select the type of sensor creating in a specific position, attached to the ego-vehicle
        self._camera_transforms = [
            (carla.Transform(carla.Location(x=-5.5, z=2.5), carla.Rotation(pitch=8.0)), Attachment.SpringArm),
            (carla.Transform(carla.Location(x=1.6, z=1.7)), Attachment.Rigid)
            ]
        self.transform_index = 1
        # Set the sensor Camera: RGB and semantic segmentation
        self.sensors = [
            ['sensor.camera.rgb', cc.Raw, 'Camera RGB', {}],
            ['sensor.camera.semantic_segmentation', cc.Raw, 'Camera Semantic Segmentation (Raw)', {}],
            ['sensor.camera.semantic_segmentation', cc.CityScapesPalette,'Camera Semantic Segmentation (CityScapes Palette)', {}]
            ]
        world = self._parent.get_world()
        bp_library = world.get_blueprint_library()
        # Set the sensor attributes of the camera
        for item in self.sensors:
            bp = bp_library.find(item[0])
            if item[0].startswith('sensor.camera'):
                bp.set_attribute('image_size_x', str(hud.dim[0]))
                bp.set_attribute('image_size_y', str(hud.dim[1]))
                if bp.has_attribute('gamma'):
                    bp.set_attribute('gamma', str(gamma_correction))
                for attr_name, attr_value in item[3].items():
                    bp.set_attribute(attr_name, attr_value)
            item.append(bp)
        self.index = None

    def toggle_camera(self):
        # Change sensor position
        self.transform_index = (self.transform_index + 1) % len(self._camera_transforms)
        self.set_sensor(self.index, notify=False, force_respawn=True)

    def set_sensor(self, index, notify=True, force_respawn=False):
        """
        ## Function used to set up the sensor
        """
        index = index % len(self.sensors)
        needs_respawn = True if self.index is None else \
            (force_respawn or (self.sensors[index][2] != self.sensors[self.index][2]))
        if needs_respawn:
            if self.sensor is not None:
                self.sensor.destroy()
                self.surface = None
            self.sensor = self._parent.get_world().spawn_actor(
                self.sensors[index][-1],
                self._camera_transforms[self.transform_index][0],
                attach_to=self._parent,
                attachment_type=self._camera_transforms[self.transform_index][1])
            
            # We need to pass the lambda a weak reference to self to avoid
            # circular reference.
            weak_self = weakref.ref(self)
            self.sensor.listen(lambda image: CameraManager._parse_image(weak_self, image))
        if notify:
            self.hud.notification(self.sensors[index][2])
        self.index = index
        
    def next_sensor(self):
        # Function used to change type of sensor
        self.set_sensor(self.index + 1)

    def toggle_recording(self):
        # Activate/Deactivate the image recording
        self.recording = not self.recording
        self.hud.notification('Recording %s' % ('On' if self.recording else 'Off'))
    
    def toggle_snapshot(self):
        # Activate/Deactivate the snapshot capture
        global j
        self.snapshot = not self.snapshot
        if self.snapshot:
            r = 'On'
        else:
            r = 'Off'
            j = 0
        self.hud.notification('Take a Snapshot %s' % r)
        
    def render(self, display):
        # Display text on the screen
        if self.surface is not None:
            display.blit(self.surface, (0, 0))

    @staticmethod
    def _parse_image(weak_self, image):
        """
        ## This function is used to manage (and save) the images coming from the camera sensor
        """
        global num
        global j
        
        self = weak_self()
        if not self:
            return

        image.convert(self.sensors[self.index][1])
        array = np.frombuffer(image.raw_data, dtype=np.dtype("uint8"))
        array = np.reshape(array, (image.height, image.width, 4))
        array = array[:, :, :3]
        array = array[:, :, ::-1]
        self.surface = pygame.surfarray.make_surface(array.swapaxes(0, 1))

        test = np.array(image.raw_data)
        test = test.copy()
        test = test.reshape((image.height, image.width, 4))
        test = test[:, :, :3]
        
        # Take the image at HD resolution (1280x720)
        new_size_img = cv2.resize(test, dsize=(1280, 720))
        
        if self.snapshot:
            # Save image snapshot on disk
            if j == 0:
                j = j + 1
                # save snapshot in the snap folder in jpg format
                cv2.imwrite('snap/input_' + str(num) + ".jpg", new_size_img)               
                num += 1

        if self.recording:
            # Save image recording on disk in _out folder in png format
            image.save_to_disk('_out/%08d' % image.frame)

# ==============================================================================
# -- game_loop() ---------------------------------------------------------------
# ==============================================================================

def game_loop(args):
    
    global all_actors
    vehicles_list = []  # instantiate the list of vehicles
    walkers_list = []   # instantiate the list of pedestrians
    all_id = []         # actors id list
    pygame.init()
    pygame.font.init()
    world = None
    
    try:
        # Connect to the client
        client = carla.Client(HOST, PORT)
        client.set_timeout(TIMEOUT) # seconds
        
        if args.map is not None:
            print('Load map %r.' % args.map)
            # Load the world for the simulation
            client.load_world(args.map)
            print(args.map + " Map Town Loaded Successfully!")
        
        # Pygame module to control the display window and screen
        display = pygame.display.set_mode(
            (args.width, args.height),
            pygame.HWSURFACE | pygame.DOUBLEBUF)
        
        # Set the head-up display (HUD)
        hud = HUD(args.width, args.height)
        # Retrieve the world
        mondo = client.get_world()
        
        # Set the world class used in the simulation
        world = World(mondo, hud, args, vehicles_list, walkers_list, all_id, client)
        
        # Set the controller keyboard class
        controller = KeyboardControl(world, args.autopilot)
        
        # Create an object to help track time
        clock = pygame.time.Clock()
        while True:
            # update the clock
            clock.tick_busy_loop(60) # milliseconds
            # check if a keyboard key has been pushed
            if controller.parse_events(client, world, clock):
                return
            world.tick(clock)
            # Display text information on the screen
            world.render(display)
            # Update the display surface to the screen
            pygame.display.flip()

    finally:
            # At the end of the simulation all the actors (vehicles and pedestrians) present in the simulation are destroyed
        if world is not None:
            print('\nDestroying all the actor created in the simulation...')
            print('\n%d vehicles destroyed' % len(vehicles_list), '+ 1 ego-vehicle destroyed')
            # destroy other vehicles
            client.apply_batch([carla.command.DestroyActor(x) for x in vehicles_list])
            # stop pedestrians (list is [controller, actor, controller, actor ...])
            for i in range(0, len(all_id), 2):
                all_actors[i].stop()
            print('\n%d walkers destroyed' % len(walkers_list))
            # destroy pedestrian (actor and controller)
            client.apply_batch([carla.command.DestroyActor(x) for x in all_id])
            world.destroy()

        pygame.quit()
        print('\nSimulation Finished.')

# ==============================================================================
# -- main() --------------------------------------------------------------------
# ==============================================================================

def main():
    argparser = argparse.ArgumentParser(
        description='CARLA Control Client')
    argparser.add_argument(
        '-v', '--verbose',
        action='store_true',
        dest='debug',
        help='print debug information')
    argparser.add_argument(
        '-f', '--fps',
        default=10,
        type=int,
        help='fps of the simulation')
    argparser.add_argument(
        '-a', '--autopilot',
        action='store_true',
        help='enable autopilot')
    argparser.add_argument(
        '--res',
        metavar='WIDTHxHEIGHT',
        default='1280x720',
        help='window resolution (default: 1280x720)')
    argparser.add_argument(
        '-m', '--map',
        default = 'Town04',
        type=str,
        help='load a map')
    argparser.add_argument(
        '-r', '--reload-map',
        action='store_true',
        help='reload current map')
    argparser.add_argument(
        '-n', '--number-of-vehicles',
        metavar='N',
        default=30,
        type=int,
        help='number of vehicles (default: 10)')
    argparser.add_argument(
        '-w', '--number-of-walkers',
        metavar='W',
        default=50,
        type=int,
        help='number of walkers (default: 50)')
    argparser.add_argument(
        '--gamma',
        default=2.2,
        type=float,
        help='Gamma correction of the camera (default: 2.2)')
    
    args = argparser.parse_args()

    args.width, args.height = [int(x) for x in args.res.split('x')]

    log_level = logging.DEBUG if args.debug else logging.INFO
    logging.basicConfig(format='%(levelname)s: %(message)s', level=log_level)

    logging.info('listening to server %s:%s', HOST, PORT)

    print(__doc__)

    try:

        game_loop(args)

    except KeyboardInterrupt:
        print('\nCancelled by user. Bye!')

if __name__ == '__main__':

    main()