package eu.europa.ec.healtheid.management;

import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Mockito;
import org.mockito.internal.matchers.Any;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.test.util.ReflectionTestUtils;
import org.springframework.ui.ModelMap;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import eu.europa.ec.healtheid.adaptor.model.HealtheidCountryConfiguration;
import eu.europa.ec.healtheid.exception.AcknowledgmentNotFoundException;
import eu.europa.ec.healtheid.exception.HealtheidGetCountryConfigurationException;
import eu.europa.ec.healtheid.exception.PatientDataNotReadyException;
import eu.europa.ec.healtheid.exception.PatientNotFoundException;
import eu.europa.ec.healtheid.models.Demographics;
import eu.europa.ec.healtheid.models.Encounter;
import eu.europa.ec.healtheid.models.NotificationData;
import eu.europa.ec.healtheid.models.PatientAcknowledge;
import eu.europa.ec.healtheid.models.PatientData;
import eu.europa.ec.healtheid.models.SearchFieldsForm;
import eu.europa.ec.healtheid.models.enumerator.NotificationType;
import eu.europa.ec.healtheid.repository.AcknowledgeRepository;
import eu.europa.ec.healtheid.repository.EncounterRepository;
import eu.europa.ec.healtheid.repository.PatientDataRepository;
import eu.europa.ec.healtheid.services.NcpHProxyService;
import eu.europa.ec.healtheid.services.PatientConnector;
import eu.europa.ec.healtheid.utils.JwtTokenUtils;

import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.PATIENT_ACKNOWLEDGE_URL;
import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.PATIENT_BASE_URL;
import static org.junit.Assert.*;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.mockito.junit.MockitoJUnitRunner;
import static org.mockito.ArgumentMatchers.any;



@RunWith(MockitoJUnitRunner.class)
public class WorkflowManagerTest {
	
	@InjectMocks
	WorkflowManager workflowmanager= new WorkflowManager();
	
    @Mock
    EncounterRepository encounterRepository;
    
    @Mock
    PatientDataRepository patientRepository;
    
    @Mock
    AcknowledgeRepository acknowledgeRepository;
    
    @Mock
    JwtTokenUtils jwtTokenUtils;
    
    @Mock
    NcpHProxyService ncphproxy;
    
    @Mock
    PatientConnector patientConnector;
     
    @Before
    public void setup () {
        ReflectionTestUtils.setField(workflowmanager, "serverUrl", "localhost");
        ReflectionTestUtils.setField(workflowmanager, "port", "8080");
        ReflectionTestUtils.setField(workflowmanager, "eidasHProxyURL", "http://heid-connector.test:8084/eidas-hproxy/authentication");
    }
    

    @Test
    public void createEncounter() {
    	Encounter encounter= new Encounter();
    	encounter.setPatientCountryCode("IT");
		encounter.setPatientEmail("test@email.com");
		encounter.setPatientMobilePhone("+393217654321");
		
 
		Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
		Mockito.when(patientConnector.sendLinkToPatient(any(String.class), any(String.class)))
					.thenReturn(true);
		
    	try{
    		encounter= workflowmanager.createEncounter(encounter);
		} catch (Exception e) {
			e.printStackTrace();
			fail(e.getMessage());
		}
    	
    	assertNotNull("the encounterID aka jwt Token cannot be null",encounter.getToken());
    	assertNotNull("Patient consent cannot be null",encounter.isConsent());
    	assertFalse("The patient consent must be set to false",encounter.isConsent());
    }

    @Test
    public void eIDASredirection() {
    	
    	String encounterID = jwtTokenUtils.createToken(new Encounter());
    	ModelMap model=new ModelMap();
    	
    	ModelAndView redirect=workflowmanager.eIDASredirection(model, encounterID);
   	 	assertNotNull("the View returned by workflow manager cannot be null",redirect);
   	 	assertEquals("The encounterID and token in the model must be equals",redirect.getModelMap().get("token"), encounterID);
    }
    
    @Test
    public void privateGenerateNCPPatientData() 
    		throws NoSuchMethodException, SecurityException, IllegalAccessException, 
    			IllegalArgumentException, InvocationTargetException {
    	
    	HashMap <String,String> eidasAttributes=new HashMap<>();
    	HashMap <String,String> map=new HashMap<>();
        	
    	eidasAttributes.put("FamilyName", "John");
    	eidasAttributes.put("FirstName",  "Snow");
    	eidasAttributes.put("DateOfBirth",  "02/03/1987");
    	eidasAttributes.put("PersonIdentifier",  "W1NT3R1SC0M1NG");
    	eidasAttributes.put("PlaceOfBirth",  "Winterfell");
    	
    	Method method =
    			WorkflowManager.class.getDeclaredMethod("generateNCPPatientData", String.class,java.util.Map.class,PatientData.class, HealtheidCountryConfiguration.class);
    	method.setAccessible(true);
    	PatientData output=(PatientData)method.invoke(workflowmanager, new String(),eidasAttributes, new PatientData(),null );	
    	output.getDemographics().forEach(d-> System.out.println("key: " + d.getFriendlyName() + "  value: "+ d.getUserValue()));
    	output.getDemographics().forEach(d-> map.put(d.getFriendlyName(), d.getUserValue() ));
    	assertNotNull(map);
    	assertEquals(eidasAttributes,map); 	
    }
    
    @Test(expected = AcknowledgmentNotFoundException.class)
    public void updateAcknlowledgement_AcknowledgementNotFound() throws AcknowledgmentNotFoundException, PatientNotFoundException {
    	
    	NotificationData notice=new NotificationData();
    	notice.setEncounterID(jwtTokenUtils.createToken(new Encounter()));
    	workflowmanager.updateAcknlowledgement(notice);
    }
       
    @Test	
    public void processEncounterData() throws HealtheidGetCountryConfigurationException {

    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");

    	String encounterID=jwtTokenUtils.createToken(encounter);
    	MockHttpServletRequest request = new MockHttpServletRequest();
    	RedirectAttributes redirectedAtributes = Mockito.mock(RedirectAttributes.class);
    	HashMap <String,String> eidasAttributes=new HashMap<>();
    	
    	eidasAttributes.put("FamilyName", "John");
    	eidasAttributes.put("FirstName",  "Snow");
    	eidasAttributes.put("DateOfBirth",  "02/03/1987");
    	eidasAttributes.put("PersonIdentifier",  "W1NT3R1SC0M1NG");
    	eidasAttributes.put("PlaceOfBirth",  "Winterfell");
    	eidasAttributes.put("TaxReference",  "T4XR3F3R3NC3C0D3");
    	
    	String redirect = workflowmanager.processEncounterData(encounterID, request, 
    			eidasAttributes, redirectedAtributes);
    	
    	assertNotNull(redirect);
    	assertEquals("redirect:/"+PATIENT_BASE_URL+PATIENT_ACKNOWLEDGE_URL+"/"+encounterID,redirect);
    }
    
    @Test
    public void acknowledgeStore() throws Exception {
    	
    	Encounter encounter=new Encounter();
    	encounter.setPatientCountryCode("IT");
		encounter.setPatientEmail("test@email.com");
		encounter.setPatientMobilePhone("+393217654321");


    	String encounterID=jwtTokenUtils.createToken(encounter);
    	String PINReference="Patient Information Notice";
    	workflowmanager.acknowledgeStore(encounterID, PINReference);
		ArgumentCaptor<PatientAcknowledge> argumentCaptor = ArgumentCaptor.forClass(PatientAcknowledge.class);

    	Mockito.verify(acknowledgeRepository, Mockito.times(1)).save(argumentCaptor.capture());

    	assertEquals(PINReference, argumentCaptor.getValue().getPIN());
    	assertEquals(encounterID, argumentCaptor.getValue().getJWTtoken());
    	assertTrue( argumentCaptor.getValue().isConsentGiven());

    }
    
    @Test(expected=AcknowledgmentNotFoundException.class)
    public void updateAcknlowledgement_AckNotFoundException() throws AcknowledgmentNotFoundException, PatientNotFoundException {
    	
    	Encounter encounter=new Encounter();
    	NotificationData notice= new NotificationData();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	
        notice.setEncounterID(encounterID);
        notice.setNotificationType(NotificationType.CONSENT);
		workflowmanager.updateAcknlowledgement(notice);
    }
    @Test
    public void updateAcknlowledgement() throws AcknowledgmentNotFoundException, PatientNotFoundException {
    	
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);

    	PatientAcknowledge consent = new PatientAcknowledge();
        consent.setConsent(true);
        consent.setJWTtoken(encounter.getToken());
        consent.setTimestamp(new Date(System.currentTimeMillis()));
        consent.setPIN("Patient Information Notice");
        
        Mockito.when(acknowledgeRepository.existsById(encounterID)).thenReturn(true);
    	Mockito.when(acknowledgeRepository.findById(encounterID)).thenReturn(Optional.of(consent));
    	
    	String patientID="This_is_the_PatientID";
    	Demographics patientDemographics = new Demographics();	
		patientDemographics.setFriendlyName("healthInsuranceId");
		patientDemographics.setUserValue(patientID);
		patientDemographics.setMandatory(true);
    	
    	PatientData patient=new PatientData();
    	patient.setEncounterID(encounterID);
    	patient.getDemographics().add(patientDemographics);
    	Mockito.when(patientRepository.existsById(encounterID)).thenReturn(true);
    	Mockito.when(patientRepository.findById(encounterID)).thenReturn(Optional.of(patient));
    	
    	NotificationData notice= new NotificationData();
        notice.setEncounterID(encounterID);
        notice.setNotificationType(NotificationType.CONSENT);
        
		workflowmanager.updateAcknlowledgement(notice);
		
		assertTrue(acknowledgeRepository.existsById(encounterID));
		
		PatientAcknowledge acknowledge= acknowledgeRepository.findById(encounterID).get();
		assertNotNull(acknowledge);
		assertNotNull(acknowledge.getPIN());
		assertNotNull(acknowledge.getTimestamp());
		assertNotNull(acknowledge.getPatientID());

		assertEquals(patientID, acknowledge.getPatientID());
    }
    
    @Test(expected = PatientNotFoundException.class)
    public void putAdditionalPatientData_PatientNotFoundException() throws PatientNotFoundException, HealtheidGetCountryConfigurationException {
    
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	
    	SearchFieldsForm searchFieldsForm = new SearchFieldsForm();
    	
    	workflowmanager.putAdditionalPatientData(encounterID,searchFieldsForm);
    }
    
    @Test
    public void putAdditionalPatientData() throws PatientNotFoundException, HealtheidGetCountryConfigurationException {
    
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	
    	Mockito.when(patientRepository.existsById(encounterID)).thenReturn(true);
    	
    	PatientData patient=new PatientData();
    	patient.setEncounterID(encounterID);
    	Mockito.when(patientRepository.findById(encounterID)).thenReturn(Optional.of(patient));
    	
    	Map<String, String> textFields=new HashMap<>();
    	textFields.put("givenName", "John");
    	textFields.put("surname", "Snow");
    	
    	
    	Map<String, String> ids=new HashMap<>();
    	ids.put("healthInsuranceId", "This_is_the_PatientID");
    	
    	SearchFieldsForm searchFieldsForm = new SearchFieldsForm();
		searchFieldsForm.setTextFields(textFields);
		searchFieldsForm.setIds(ids);
		
    	workflowmanager.putAdditionalPatientData(encounterID,searchFieldsForm);
    	
    	assertTrue(patientRepository.existsById(encounterID));
    	
    	PatientData patientStored= patientRepository.findById(encounterID).get();
		assertNotNull(patientStored);
		assertNotNull(patientStored.getDemographics());
		assertNotNull(patientStored.getEncounterID());
		
		String surname = " "; 
		String name= " ";
		String patientID=" ";
		List<Demographics> demographics = patientStored.getDemographics();
		
		for(int i=0; i< demographics.size();i++) {
		
			if(demographics.get(i).getFriendlyName().equals("surname"))
				surname=demographics.get(i).getUserValue();
			
			if(demographics.get(i).getFriendlyName().equals("givenName"))
				name=demographics.get(i).getUserValue();
			
			if(demographics.get(i).getFriendlyName().equals("healthInsuranceId"))
				patientID=demographics.get(i).getUserValue();						
		}
		
		assertEquals(patientID, "This_is_the_PatientID");
		assertEquals(name, "John");
		assertEquals(surname, "Snow");
    }
    
    @Test(expected = PatientDataNotReadyException.class)
    public void checkIfPatientIsAuthenticated_PatientDataNotReady() throws PatientDataNotReadyException, PatientNotFoundException {
    	
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	
    	Demographics patientDemographics = new Demographics();	
		patientDemographics.setFriendlyName("healthInsuranceId");
		patientDemographics.setUserValue("This_is_the_PatientID");
		patientDemographics.setMandatory(true);
		
		PatientData patient = new PatientData();
		patient.setEncounterID(encounterID);
		patient.getDemographics().add(patientDemographics);
		patient.setDataReady(false);
		
		Mockito.when(patientRepository.existsById(encounterID)).thenReturn(true);
		Mockito.when(patientRepository.findById(encounterID)).thenReturn(Optional.of(patient));
		
		workflowmanager.checkIfPatientIsAuthenticated(patient.getEncounterID());
    }
    
    @Test(expected = PatientNotFoundException.class)
    public void checkIfPatientIsAuthenticated_PatientNotFound() throws PatientDataNotReadyException, PatientNotFoundException {
    	
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	
		Mockito.when(patientRepository.existsById(encounterID)).thenReturn(false);
		
		workflowmanager.checkIfPatientIsAuthenticated(encounterID);
    }
    
    @Test
    public void checkIfPatientIsAuthenticated_PatientDataReady() throws PatientDataNotReadyException, PatientNotFoundException {
    	
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	
    	Demographics patientDemographics = new Demographics();	
		patientDemographics.setFriendlyName("healthInsuranceId");
		patientDemographics.setUserValue("This_is_the_PatientID");
		patientDemographics.setMandatory(true);
		
		PatientData patient = new PatientData();
		patient.setEncounterID(encounterID);
		patient.getDemographics().add(patientDemographics);
		patient.setDataReady(true);
		
		Mockito.when(patientRepository.existsById(encounterID)).thenReturn(true);
		Mockito.when(patientRepository.findById(encounterID)).thenReturn(Optional.of(patient));
		
		workflowmanager.checkIfPatientIsAuthenticated(patient.getEncounterID());
		
		PatientData patientStored = patientRepository.findById(encounterID).get();
		assertNotNull(patientStored);
		assertEquals(patientStored.getEncounterID(), patient.getEncounterID());
		assertEquals(patientStored.getDemographics(), patient.getDemographics());
    }
    
    @Test
    public void notifyPatientByEmail() throws Exception {
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	String patientEmail="email@test.it";
    	encounter.setPatientEmail(patientEmail);
    	
    	Mockito.when(encounterRepository.findById(encounterID)).thenReturn(Optional.of(encounter));
    	Mockito.when(
    			patientConnector.sendNotificationToPatient(NotificationType.CONSENT,patientEmail))
    			.thenReturn(true);
    	
    	NotificationData notice= new NotificationData();
        notice.setEncounterID(encounterID);
        notice.setNotificationType(NotificationType.CONSENT);
        
        boolean notified=workflowmanager.notifyPatient(notice);
        
        assertTrue(notified);
        
    }
    
    @Test
    public void notifyPatientByPhone() throws Exception {
    	Encounter encounter=new Encounter();
    	Mockito.when(jwtTokenUtils.createToken(encounter)).thenReturn("aaa.bbb.ccc");
    	String encounterID=jwtTokenUtils.createToken(encounter);
    	String phone="1234567890";
    	encounter.setPatientMobilePhone(phone);
    	
    	Mockito.when(encounterRepository.findById(encounterID)).thenReturn(Optional.of(encounter));
    	Mockito.when(
    			patientConnector.sendNotificationToPatient(NotificationType.CONSENT,phone))
    			.thenReturn(true);
    	
    	NotificationData notice= new NotificationData();
        notice.setEncounterID(encounterID);
        notice.setNotificationType(NotificationType.CONSENT);
        
        boolean notified=workflowmanager.notifyPatient(notice);
        
        assertTrue(notified);
        
    }
   
}
