package eu.europa.ec.healtheid.controllers;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.autoconfigure.web.servlet.WebMvcTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.result.MockMvcResultHandlers;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;
import org.springframework.ui.ModelMap;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import com.fasterxml.jackson.databind.ObjectMapper;

import eu.europa.ec.healtheid.adaptor.model.HealtheidCountryConfiguration;
import eu.europa.ec.healtheid.management.WorkflowManager;
import eu.europa.ec.healtheid.models.Encounter;
import eu.europa.ec.healtheid.models.NotificationData;
import eu.europa.ec.healtheid.models.SearchFieldsForm;
import eu.europa.ec.healtheid.repository.AcknowledgeRepository;
import eu.europa.ec.healtheid.repository.EncounterRepository;
import eu.europa.ec.healtheid.repository.PatientDataRepository;
import eu.europa.ec.healtheid.security.JwtAuthenticationEntryPoint;
import eu.europa.ec.healtheid.services.NcpHProxyService;
import eu.europa.ec.healtheid.services.PatientConnector;
import eu.europa.ec.healtheid.utils.FileUtils;
import eu.europa.ec.healtheid.utils.JwtTokenUtils;

import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.*;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.redirectedUrl;


import java.util.Optional;

import org.junit.Before;
import org.junit.Ignore;

@RunWith(SpringRunner.class)
@WebMvcTest(controllers = PatientController.class)
public class PatientControllerTest {
	
	@SpyBean
    WorkflowManager workflowManager;

    @MockBean
    NcpHProxyService ncphproxy;

    @MockBean
    PatientConnector patientConnector;

    @MockBean
    AcknowledgeRepository acknowledgeRepository;

    @MockBean
    PatientDataRepository patientRepository;

    @MockBean
    EncounterRepository encounterRepository;

    @MockBean
	JwtAuthenticationEntryPoint unauthorizedHandler;
	
	@MockBean
	JwtTokenUtils jwtTokenUtils;
	
	@MockBean
	FileUtils fileUtils;
	
	@Autowired
	private MockMvc mockMvc;
	
	@Value("${redirect.to.eidasHProxy}")
    private String eidasHProxyURL;
	
	@Value("${server.url}")
    private String serverUrl;
    
	@Value("${server.port}")
    private String port;
	
	@Before
	public void setup () {
		when(jwtTokenUtils.validateToken(any(String.class))).thenReturn(true);
		when(jwtTokenUtils.createToken(any(Encounter.class))).thenReturn("aaa.bbb.ccc");
		when(encounterRepository.findById(any(String.class))).thenReturn(Optional.of(new Encounter()));
	}

    @Test
    public void acceptEncounter() throws Exception {
    	
    	String token = "aaa.bbb.ccc";
    	String viewURL="redirect:"+eidasHProxyURL;
    	
    	String url=  PATIENT_BASE_URL + PATIENT_ACCEPT_ENCOUNTER_URL;
    	
    	mockMvc.perform(get(url,token)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
		.andExpect(MockMvcResultMatchers.view().name(viewURL));
    	
    	Mockito.verify(workflowManager, times(1)).eIDASredirection(any(ModelMap.class),any(String.class));
    }

    @Test
    public void additionalPatientData() throws Exception {
    	
    	String token = "aaa.bbb.ccc";
		
    	when(jwtTokenUtils.readCountry(any(String.class))).thenReturn("PT");

    	when(ncphproxy.getCountryConfiguration(any(String.class)))
							.thenReturn(any(HealtheidCountryConfiguration.class));
    	
    	String url=  PATIENT_BASE_URL + PATIENT_DATA_URL;
    	
    	mockMvc.perform(get(url)
    			.flashAttr("token", token)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE));
  		   
    	Mockito.verify(workflowManager, times(1)).getCountryConfiguration(any(String.class));
    }

    @Test
    public void postAdditionalPatientData_SearchFieldsFormIsNotValid() throws Exception {
    	String token = "aaa.bbb.ccc";
    	SearchFieldsForm searchFieldsForm=new SearchFieldsForm();
    	
    	
    	String url=  PATIENT_BASE_URL + PATIENT_DATA_POST_URL;
    	
		mockMvc.perform(post(url)
    			.flashAttr("token", token)
    			.flashAttr("searchFieldsForm", searchFieldsForm)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE));
		
    	Mockito.verify(workflowManager, times(1)).getCountryConfiguration(any(String.class));    		   
    }

    @Test
    public void patientAcknowledge_expetedAcknowledgeView() throws Exception {
    	String token = "aaa.bbb.ccc";
    	Encounter encounter=new Encounter();
    	encounter.setToken(token);
    	encounter.setConsent(false);
    	
    	when(fileUtils.loadFileAsString(any(String.class))).thenReturn(new String());
    	
    	String url=  PATIENT_BASE_URL + PATIENT_ACKNOWLEDGE_URL;
    	
    	mockMvc.perform(get(url)
    			.flashAttr("token", token)
    			.flashAttr("encounter", encounter)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    	.andExpect(MockMvcResultMatchers.view().name("acknowledge"))
    	;		   
    }
    
    @Test
    public void patientAcknowledge_expetedAcknowledgeResultView() throws Exception {
    	String token = "aaa.bbb.ccc";
    	Encounter encounter=new Encounter();
    	encounter.setToken(token);
    	encounter.setConsent(true);
    	
    	when(fileUtils.loadFileAsString(any(String.class))).thenReturn(new String());
    	
    	String url=  PATIENT_BASE_URL + PATIENT_ACKNOWLEDGE_URL;
    	
    	mockMvc.perform(get(url)
    			.flashAttr("token", token)
    			.flashAttr("encounter", encounter)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    	.andExpect(MockMvcResultMatchers.view().name("acknowledgeResult"))
    	;		   
    }

    @Test
    public void acknowledge() throws Exception {
    	String token = "aaa.bbb.ccc";
    	Encounter encounter=new Encounter();
    	encounter.setToken(token);
    	encounter.setConsent(true);
    	
    	String url=  PATIENT_BASE_URL + PATIENT_ACKNOWLEDGE_POST_URL;
    	
    	mockMvc.perform(post(url)
    			.flashAttr("encounter", encounter)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    	.andExpect(MockMvcResultMatchers.view().name("acknowledgeResult"))
    	;
    		   
    	Mockito.verify(workflowManager, times(1)).acknowledgeStore(any(String.class),any(String.class));
    	Mockito.verify(workflowManager, times(1)).notifyPatient(any(NotificationData.class));

    }
}

