package eu.europa.ec.healtheid.controllers;

import com.fasterxml.jackson.databind.ObjectMapper;
import eu.europa.ec.healtheid.management.WorkflowManager;
import eu.europa.ec.healtheid.models.Demographics;
import eu.europa.ec.healtheid.models.Encounter;
import eu.europa.ec.healtheid.models.NotificationData;
import eu.europa.ec.healtheid.models.PatientAcknowledge;
import eu.europa.ec.healtheid.models.PatientData;
import eu.europa.ec.healtheid.models.enumerator.NotificationType;
import eu.europa.ec.healtheid.repository.AcknowledgeRepository;
import eu.europa.ec.healtheid.repository.EncounterRepository;
import eu.europa.ec.healtheid.repository.PatientDataRepository;
import eu.europa.ec.healtheid.security.JwtAuthenticationEntryPoint;
import eu.europa.ec.healtheid.services.NcpHProxyService;
import eu.europa.ec.healtheid.services.PatientConnector;
import eu.europa.ec.healtheid.utils.JwtTokenUtils;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.WebMvcTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.test.web.servlet.MockMvc;

import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import java.util.Optional;


@RunWith(SpringRunner.class)
@WebMvcTest(controllers = HeidClientController.class)
public class HeidClientControllerTest {
	
	@SpyBean
    WorkflowManager workflowManager;

    @MockBean
    NcpHProxyService ncphproxy;

    @MockBean
    PatientConnector patientConnector;

    @MockBean
    AcknowledgeRepository acknowledgeRepository;

    @MockBean
    PatientDataRepository patientRepository;

    @MockBean
    EncounterRepository encounterRepository;


    @MockBean
	JwtAuthenticationEntryPoint unauthorizedHandler;
	
	@MockBean
	JwtTokenUtils jwtTokenUtils;

	@Autowired
	private MockMvc mockMvc;
	
	
	@Before
	public void setup () {
		when(jwtTokenUtils.validateToken(any(String.class))).thenReturn(true);
		when(jwtTokenUtils.createToken(any(Encounter.class))).thenReturn("aaa.bbb.ccc");
		when(encounterRepository.findById(any(String.class))).thenReturn(Optional.of(new Encounter()));
    }
		
	@Test
    public void createEncounter() throws Exception {
    	Encounter encounterReq= new Encounter();
    	encounterReq.setPatientCountryCode("IT");
    	encounterReq.setPatientMobilePhone("+39 321 1234567");
    	encounterReq.setPatientEmail("email@test.com");
    	
    	String url=  HEID_CONNECTOR_BASE_URL + HEID_CONNECTOR_CREATE_ENCOUNTER_URL;

    	mockMvc.perform(post(url)
                .secure(true)
    		    .contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
    			.content(asJsonString(encounterReq))
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    		    .andExpect(status().isCreated())
    		    .andExpect(content().contentType
                        (MediaType.APPLICATION_JSON_UTF8_VALUE));

    	Mockito.verify(workflowManager, times(1)).createEncounter(any(Encounter.class));
    }

    @Test
    public void requestPatientAuthNData_DataIsReady() throws Exception {
		String encounterID="aaa.bbb.ccc";
		PatientData patient=new PatientData();
		patient.setDataReady(true);
		patient.setEncounterID(encounterID);
		
		when(patientRepository.existsById(any(String.class))).thenReturn(true);
		when(patientRepository.findById(any(String.class))).thenReturn(Optional.of(patient));
			
		String url=  HEID_CONNECTOR_BASE_URL + HEID_CONNECTOR_PATIENT_DATA_URL;
		
    	mockMvc.perform(post(url)
    			.secure(true)
    			.contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
    			.header("Authorization", encounterID)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    		    .andExpect(status().isOk()) //expected 200 OK (Data is ready)
    		    .andExpect(content().contentType
                        (MediaType.APPLICATION_JSON_UTF8_VALUE));
    	
    	Mockito.verify(workflowManager, times(1)).checkIfPatientIsAuthenticated(any(String.class));
    }
	
    @Test
    public void requestPatientAuthNData_DataIsNotReady() throws Exception {
		String encounterID="aaa.bbb.ccc";
		PatientData patient=new PatientData();
		patient.setDataReady(false);
		patient.setEncounterID(encounterID);
		
		when(patientRepository.existsById(any(String.class))).thenReturn(true);
		when(patientRepository.findById(any(String.class))).thenReturn(Optional.of(patient));
			
		String url=  HEID_CONNECTOR_BASE_URL + HEID_CONNECTOR_PATIENT_DATA_URL;
		
    	mockMvc.perform(post(url)
    			.secure(true)
    			.contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
    			.header("Authorization", encounterID)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    		    .andExpect(status().isNoContent()); //expected 204 NO CONTENT (Data is not ready)
    		   
    	Mockito.verify(workflowManager, times(1)).checkIfPatientIsAuthenticated(any(String.class));
    }
	
    @Test
    public void requestPatientAuthNData_PatientNotFound() throws Exception {
		String encounterID="aaa.bbb.ccc";
		PatientData patient=new PatientData();
		patient.setDataReady(false);
		patient.setEncounterID(encounterID);
		
		when(patientRepository.existsById(any(String.class))).thenReturn(false);
				
		String url=  HEID_CONNECTOR_BASE_URL + HEID_CONNECTOR_PATIENT_DATA_URL;
		
    	mockMvc.perform(post(url)
    			.secure(true)
    			.contentType(MediaType.APPLICATION_JSON_UTF8_VALUE)
    			.header("Authorization", encounterID)
    			.accept(MediaType.APPLICATION_JSON_UTF8_VALUE))
    		    .andExpect(status().is5xxServerError()) //expected 500 Internal Error
    		    .andExpect(content().contentType
                        (MediaType.APPLICATION_JSON_UTF8_VALUE));
    	
    	Mockito.verify(workflowManager, times(1)).checkIfPatientIsAuthenticated(any(String.class));
    }

    @Test
    public void receiveNotice_with_UdateACK() throws Exception {
    	
		String encounterID="aaa.bbb.ccc";
    	NotificationData notice= new NotificationData();
    	notice.setEncounterID(encounterID);
    	notice.setNotificationType(NotificationType.XCPD);
    	
    	PatientAcknowledge patientAck = new PatientAcknowledge();
    
    	Demographics dem = new Demographics();
    	dem.setFriendlyName(PATIENT_ID_FRIENDLY_NAME);
    	dem.setUserValue("This is the PatientID");
    	PatientData patient = new PatientData();
    	patient.getDemographics().add(dem);
    

		when(acknowledgeRepository.existsById(any(String.class))).thenReturn(true);
		when(acknowledgeRepository.findById(any(String.class))).thenReturn(Optional.of(patientAck));
		
		when(patientRepository.existsById(any(String.class))).thenReturn(true);
		when(patientRepository.findById(any(String.class))).thenReturn(Optional.of(patient));
    	
    	String url=  HEID_CONNECTOR_BASE_URL+HEID_CONNECTOR_PATIENT_NOTIFY_URL;
    	
		mockMvc.perform(post(url)
    			.secure(true)
    		    .contentType(MediaType.APPLICATION_JSON)
    		    .content(asJsonString(notice))
    		    .header("Authorization", encounterID)
    			.accept(MediaType.APPLICATION_JSON))
    		    .andExpect(status().isOk());
    	
    	Mockito.verify(workflowManager, times(1)).updateAcknlowledgement(any(NotificationData.class));
    	Mockito.verify(workflowManager, times(1)).notifyPatient(any(NotificationData.class));

    }
    
    @Test
    public void receiveNotice() throws Exception {
    	
		String encounterID="aaa.bbb.ccc";
    	NotificationData notice= new NotificationData();
    	notice.setEncounterID(encounterID);
    	notice.setNotificationType(NotificationType.EP);
    		
    	String url=  HEID_CONNECTOR_BASE_URL+HEID_CONNECTOR_PATIENT_NOTIFY_URL;
    	
		mockMvc.perform(post(url)
    			.secure(true)
    		    .contentType(MediaType.APPLICATION_JSON)
    		    .content(asJsonString(notice))
    		    .header("Authorization", encounterID)
    			.accept(MediaType.APPLICATION_JSON))
    		    .andExpect(status().isOk());
    	
    	Mockito.verify(workflowManager, times(0)).updateAcknlowledgement(any(NotificationData.class));
    	Mockito.verify(workflowManager, times(1)).notifyPatient(any(NotificationData.class));

    }
    
    public static String asJsonString(final Object obj) {
        try {
            final ObjectMapper mapper = new ObjectMapper();
            final String jsonContent = mapper.writeValueAsString(obj);
            System.out.println(jsonContent);
            return jsonContent;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
