package eu.europa.ec.healtheid.utils;

import eu.europa.ec.healtheid.models.Encounter;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import javax.crypto.spec.SecretKeySpec;
import java.security.Key;
import java.util.Date;
import java.util.UUID;

@Component
public class JwtTokenUtils {

	private static final Logger LOGGER = LoggerFactory.getLogger(JwtTokenUtils.class);

	@Value("${jwt.secret}")
	private String secret;

	@Value("${jwt.expiration}")
	private Long expiration;

	/**
	 * Method responsible for jwt token validation
	 *
	 * @param token JWT token as String
	 *
	 * @return boolean (true = jwt token is valid)
	 */
	public boolean validateToken(String token) {

		LOGGER.trace("start validateToken method");

		LOGGER.info("validate the jwt Token");
		byte[] secret256hex = DigestUtils.sha256(secret);
		Claims claims;
		try {
			// This line will throw an exception if token is not a signed JWS (as expected)
			claims = Jwts.parser().setSigningKey(secret256hex).parseClaimsJws(token).getBody();
			LOGGER.info("Expiration time: " + claims.getExpiration());
		} catch (Exception e) {
			LOGGER.info("Invalid JWT token. An exception occurs parsing the JWT or checking the signature: ",
					e.getMessage());
			return false;
		}
		final Date expiration = claims.getExpiration();
		return expiration.after(new Date());
	}

	/**
	 * Method responsible for create jwt token
	 * @param encounterRequest
	 * @return new token
	 */
	public String createToken(Encounter encounterRequest) {
		
		LOGGER.trace("JwtTokenUtils - createToken ");
		
		final String uuid = UUID.randomUUID().toString();
		byte[] secret256hex = DigestUtils.sha256(secret);
		SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS256;
		Key signingKey = new SecretKeySpec(secret256hex, signatureAlgorithm.getJcaName());

		LOGGER.info("Start to build JWT token");
		String jwtToken = Jwts.builder().setId(uuid)
				.setExpiration(generateExpirationDate())
				.setSubject(encounterRequest.getPatientCountryCode())
				// .setClaims(claims)
				.signWith(signingKey).compact();
		
		LOGGER.debug("Token created: " + jwtToken);
		LOGGER.trace("JwtTokenUtils - createToken ");
		
		return jwtToken;
	}

	private Date generateExpirationDate() {
		return new Date(System.currentTimeMillis() + expiration * 1000);
	}

	/**
	 * This method reads the Patient Country from the jwt token
	 * 
	 * @param  token
	 * 
	 * @return Patient Country
	 */
	public String readCountry(String token) {
		
		LOGGER.trace("readCountry");
		
		LOGGER.info("retrive country from jwt token");
		byte[] secret256hex; 
		Claims claims;
		String country;
		  
		secret256hex= DigestUtils.sha256(secret);
		
		claims = Jwts.parser()         
			       .setSigningKey(secret256hex)
			       .parseClaimsJws(token).getBody();		
		
		country=claims.getSubject(); 
		
		LOGGER.debug("Patient country: " + country);
		LOGGER.trace("readCountry");
		
		return country;
	}
}
