package eu.europa.ec.healtheid.management;

import eu.europa.ec.healtheid.adaptor.model.HealtheidCountryConfiguration;

import eu.europa.ec.healtheid.exception.AcknowledgmentNotFoundException;
import eu.europa.ec.healtheid.exception.HealtheidGetCountryConfigurationException;
import eu.europa.ec.healtheid.exception.PatientDataNotReadyException;
import eu.europa.ec.healtheid.exception.PatientNotFoundException;
import eu.europa.ec.healtheid.models.*;
import eu.europa.ec.healtheid.repository.AcknowledgeRepository;
import eu.europa.ec.healtheid.repository.EncounterRepository;
import eu.europa.ec.healtheid.repository.PatientDataRepository;
import eu.europa.ec.healtheid.services.NcpHProxyService;
import eu.europa.ec.healtheid.services.PatientConnector;
import eu.europa.ec.healtheid.utils.JwtTokenUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.ui.ModelMap;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.Date;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletRequest;
import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.*;

@Component
public class WorkflowManager {

    private static final Logger LOGGER = LoggerFactory.getLogger(WorkflowManager.class);

    @Autowired
    NcpHProxyService ncphproxy;
    
    @Autowired
    PatientConnector patientConnector;

    @Autowired
    JwtTokenUtils jwtTokenUtils;

    @Autowired
    AcknowledgeRepository acknowledgeRepository;
    
    @Autowired
    PatientDataRepository patientRepository;
    
    @Autowired
    EncounterRepository encounterRepository;

    @Value("${server.url}")
    private String serverUrl;
    
    @Value("${server.port}")
    private String port;
    
    @Value("${server.protocol}")
    private String protocol;

    private final String patientEndpoint = "/healtheid-connector/patientEncounter/acceptEncounter/";

    @Value("${redirect.to.eidasHProxy}")
    private String eidasHProxyURL;


    /**
     * Method responsible to create the encounter
     * After the JWT generation, this method create the link to send towards the Patient
     * Note that "serverUrl" and "port" are configurable from default.properties
     *
     * @param encounterRequest - object that represent the request coming from HeiD Client
     * @return encounter object filled with the encounterID (JSON Web Token)
     */
    public Encounter createEncounter(Encounter encounterRequest) {
    	
    	LOGGER.trace("start createEncounter method");
        
    	// 1. Create tokenSecurity
        String jwtToken = jwtTokenUtils.createToken(encounterRequest);
        
        // 2. Send Link to Patient;
        UriComponents uri = UriComponentsBuilder.newInstance()
                .scheme(protocol).host(serverUrl).port(port)
                .path(patientEndpoint+jwtToken).build();
        LOGGER.debug("Link to send to patient... " + uri.toString());
        boolean sentLink=false;
        if (encounterRequest.getPatientEmail() != null)
        	sentLink = patientConnector.sendLinkToPatient(uri.toString(), encounterRequest.getPatientEmail());
        if (encounterRequest.getPatientMobilePhone() != null)
        	sentLink = patientConnector.sendLinkToPatient(uri.toString(), encounterRequest.getPatientMobilePhone());
        
        LOGGER.debug("Has the link been sent correctly? " + sentLink);
        
        encounterRequest.setToken(jwtToken);
        encounterRequest.setConsent(false);
        encounterRepository.save(encounterRequest);
        
        LOGGER.trace("createEncounter");

        return encounterRequest;
    }
    
    /**
     * Method responsible for returning the country configuration
     *
     * @param jwt token -  token that contains the country value  
     * @return HealtheidCountryConfiguration - the country configuration provided by OpenNCP
     * @throws HealtheidGetCountryConfigurationException 
     */
    public HealtheidCountryConfiguration getCountryConfiguration(String token) throws HealtheidGetCountryConfigurationException { 	
    	String country = jwtTokenUtils.readCountry(token);
    	return ncphproxy.getCountryConfiguration(country);
    }

    /**
     * Method responsible for storing the Acknowledge in DB
     *
     * @param jwtToken - encounterID that uniquely identifies the encounter between HP and Patient
     * @param PINReference - the version of Patient Information Notice showed to Patient
     * @throws Exception - Internal Error during notifyPatient() method
     * 						
     */
    public void acknowledgeStore(String jwtToken, String PINReference) throws Exception {

        PatientAcknowledge consent = new PatientAcknowledge();
        consent.setConsent(true);
        consent.setJWTtoken(jwtToken);
        consent.setTimestamp(new Date(System.currentTimeMillis()));
        consent.setPIN(PINReference);
        acknowledgeRepository.save(consent);
    }

    /**
     * Method responsible to store eidas attributes of the patient
     * in the PatientRepository
     *
     * @param patientData - object that contains the Patient attributes
     * 
     */
    public void storePatientData(PatientData patientData) {
        patientRepository.save(patientData);
    }

    /**
     * Method responsible to check if patient has completed the eIDAS AuthN
     * Note that this method is called when the HeiD Client polls the HeiD Connector
     *
     * @param encounterID (aka JSON Web Token) encounterID that uniquely identifies the encounter
     * @return PatientData if patient is authenticated and necessary all data are present
     * @throws PatientNotFoundException     - encounterID is valid but belongs to no one
     * @throws PatientDataNotReadyException - if PatientData is not ready
     */
    public PatientData checkIfPatientIsAuthenticated(String encounterID) throws PatientDataNotReadyException, PatientNotFoundException {
    	
    	LOGGER.trace("start checkIfPatientIsAuthenticated method");
    	
    	LOGGER.debug("check patient existance. EncounterID:" + encounterID);
    	
        if (!patientRepository.existsById(encounterID))
            throw new PatientNotFoundException("This encounterID is valid but unknown. "
                    + "Probably the Patient has not yet clicked on the link.");
        
        PatientData patient = new PatientData();
        patient = patientRepository.findById(encounterID).get();
        LOGGER.info("The encounter exist. Check if patient has completed the eIDAS authentication process");
        if (!patient.dataIsReady())
            throw new PatientDataNotReadyException("Patient AuthN process is not complete. Please, try later.");

    	LOGGER.info("Patient data are in DB.");
        return patient;
    }

    /**
     * This method will update the patient data stored in the PatientRepository
     *
     * @param patientData
     * @throws PatientNotFoundException - the Patient not exist in DB
     */
    public void updatePatientData(PatientData patientData) throws PatientNotFoundException {
    	patientRepository.save(patientData);
    }


    /**
     * Method responsible to send notification to Patient
     * The notice action will be delegated to the HealtheID Notification component
     *
     * @param notice - object that defines the kind of notice
     * @throws Exception - retrieving patientData from DB or sending the notification
     */
    public boolean notifyPatient(NotificationData notice) throws Exception {
    	
    	LOGGER.trace("start notifyPatient method");
    	LOGGER.debug("notify patient about: " + notice.getNotificationType());
    	//access to DB to retrive patient email or phone    	
    	Encounter encounter= encounterRepository.findById(notice.getEncounterID()).get();
    	boolean notified=false;
        if (encounter.getPatientEmail() != null)
        	notified= patientConnector.sendNotificationToPatient(notice.getNotificationType(), encounter.getPatientEmail());
          
        if (encounter.getPatientMobilePhone() != null)
        	notified= patientConnector.sendNotificationToPatient(notice.getNotificationType(), encounter.getPatientMobilePhone());
         
        if (!notified)
            LOGGER.info("Unable to send the notification");
        else
        	LOGGER.info("Notification sent to patient");
        
        return notified;
    }

    /**
     * This method redirect Patient towards eidasHProxy to perform eIDAS AuthN
     * @param encounterID - token that uniquely identifies the encounter
     *
     * @return redirect to eidas
     */
    public ModelAndView eIDASredirection(ModelMap model, String encounterID) {
    	
    	LOGGER.trace("start eIDASredirection method");
 
    	PatientData patientData= new PatientData();
    	patientData.setEncounterID(encounterID);
    	storePatientData(patientData);
    	
    	model.addAttribute("token", encounterID);
    	
    	LOGGER.info("Redirecting the patient to eIDAS HProxy...");
    	
        return new ModelAndView("redirect:" + eidasHProxyURL, model);
    }
    
    /**
     * This method handles the eidas data set and trasforms data from eidas format in Open NCP format
     * 
     * @param encounterID - token that uniquely identifies the encounter
     *
     * @return redirect the Patient to add some data (if needed)
     * 		   otherwise it redirect the patient to Consent and Acknowledge screen
     */
    public String processEncounterData(String token, HttpServletRequest request, Map<String, String> eidasAttributes, RedirectAttributes redirectAttributes) 
    		throws HealtheidGetCountryConfigurationException {
    	
    	LOGGER.trace("start processEncounterData method");

		HealtheidCountryConfiguration healtheidCountryConfiguration = getCountryConfiguration(token);
    	
		PatientData patientData= new PatientData();
		patientData.setEncounterID(token);
		
		LOGGER.info("Patient eIDAS attribute received.");
		LOGGER.info("Start to map eIDAS attributes with healtheidCountryConfiguration.");	
		patientData=generateNCPPatientData(token, eidasAttributes, patientData, healtheidCountryConfiguration);
		LOGGER.info("eIDAS attributes processed correctly.");
		
		storePatientData(patientData);
				
		request.setAttribute(View.RESPONSE_STATUS_ATTRIBUTE, HttpStatus.TEMPORARY_REDIRECT);
		redirectAttributes.addFlashAttribute("healtheidCountryConfiguration", healtheidCountryConfiguration);
		redirectAttributes.addFlashAttribute("token",token);
		
		if (healtheidCountryConfiguration != null && !healtheidCountryConfiguration.getPatientInput().isPatientInputNeeded())
			return "redirect:/"+PATIENT_BASE_URL+PATIENT_DATA_URL;
		
		return "redirect:/"+PATIENT_BASE_URL+PATIENT_ACKNOWLEDGE_URL+"/" + token;		
		
    }

    private PatientData generateNCPPatientData(String token, Map<String, String> mapAttributes, PatientData patientData, HealtheidCountryConfiguration healtheidCountryConfiguration) {
    	
		for (Entry<String, String> entry : mapAttributes.entrySet()){
			
			Demographics patientDemographics = new Demographics();
			if(entry.getKey().contentEquals("TaxReference"))
				patientData.setDataReady(true);
			patientDemographics.setFriendlyName(entry.getKey());
			patientDemographics.setUserValue(entry.getValue());
			patientDemographics.setMandatory(true);
			
			// fill others fields in patientDemographics
			if(healtheidCountryConfiguration!=null)
				healtheidCountryConfiguration.getPatientSearch().getCountry()
					.getSearchFields().getTextField().stream()
					.forEach( field -> {						
						if(field.getFriendlyName().equals(patientDemographics.getFriendlyName())) {
							patientDemographics.setLabel(field.getLabel());
							patientDemographics.setLength(field.getMax().intValue());
						}						
					});			
			patientData.getDemographics().add(patientDemographics);
		}
		
		return patientData;
	}

    /**
     * This method add news information to patientData after the patient manual input
     *
     * @param encounterID - token that uniquely identifies the encounter
     * @param searchFieldsForm
     * 
     * @throws PatientNotFoundException   The patient is not present in DB
     * @throws HealtheidGetCountryConfigurationException 
     */
    public void putAdditionalPatientData(String encounterID, SearchFieldsForm searchFieldsForm) 
    		throws PatientNotFoundException, HealtheidGetCountryConfigurationException {

    	LOGGER.trace("start putAdditionalPatientData method");
        if (!patientRepository.existsById(encounterID))
            throw new PatientNotFoundException("This encounterID is valid, but there is no Patient in DB"); 

        PatientData patient = new PatientData();
        patient = patientRepository.findById(encounterID).get();
        
		HealtheidCountryConfiguration healtheidCountryConfiguration = getCountryConfiguration(encounterID);
        //update patient.getDemographics() with TextFields in AdditionalDataForm 
		LOGGER.info("Map the data added by hand in compliance with the healtheidCountryConfiguration");
        patient = generateNCPPatientData(encounterID, searchFieldsForm.getTextFields(), patient, healtheidCountryConfiguration);
        
        LOGGER.info("Update Patient data");
        //update patient.getDemographics() with ids in AdditionalDataForm 
        for (Entry<String, String> entry : searchFieldsForm.getIds().entrySet()){
			
			Demographics patientDemographics = new Demographics();	
			patientDemographics.setFriendlyName(entry.getKey());
			patientDemographics.setUserValue(entry.getValue());
			patientDemographics.setMandatory(true);
			
			// fill others fields in patientDemographics
			if(healtheidCountryConfiguration!=null)
				healtheidCountryConfiguration.getPatientSearch().getCountry()
					.getSearchFields().getId().stream()
					.forEach( id -> {						
						if(id.getFriendlyName().equals(patientDemographics.getFriendlyName())) {
							patientDemographics.setLabel(id.getLabel());
							patientDemographics.setLength(id.getMax().intValue());
						}						
					});			
			patient.getDemographics().add(patientDemographics);
		}
        
        patient.setDataReady(true);
        updatePatientData(patient);
        LOGGER.info("All data are ok. Patient is ready to go on OpenNCP.");
    }

    /**
     * Checks the JWT signature and the JWT Expiration Date
     *
     * @param token the json web token to check
     * @return true if token is valid
     * @return false if token is invalid
     */
    public boolean validateToken(String token) {
        return jwtTokenUtils.validateToken(token);
    }
    
    /**
     * This method update the Acknowledgment in DB after the XCPD  
     *
     * @param notice - object that specifies the kind of notice with the encounterID
     * @throws AcknowledgmentNotFoundException if the Ack data structure is not in the DB
     * @throws PatientNotFoundException 
     */
	public void updateAcknlowledgement(NotificationData notice) throws AcknowledgmentNotFoundException, PatientNotFoundException {
		
		LOGGER.trace("start updateAcknlowledgement method");
		
		if (!acknowledgeRepository.existsById(notice.getEncounterID()))
			throw new AcknowledgmentNotFoundException("Cannot update the Acknowlege. Not found in DB.");
		
		if (!patientRepository.existsById(notice.getEncounterID()))
            throw new PatientNotFoundException("Encounter not found in patientRepository");
		
		PatientAcknowledge consent = new PatientAcknowledge();
		PatientData patient = new PatientData();
		
		consent = acknowledgeRepository.findById(notice.getEncounterID()).get();
	    patient = patientRepository.findById(notice.getEncounterID()).get();
		
	    LOGGER.info("Try to retrive the patientID from DB.");
	    String patientID=
	    		patient.getDemographics().stream()
	    				.filter(demographic -> demographic.getFriendlyName().equals(PATIENT_ID_FRIENDLY_NAME))
	    				.findFirst()
	    				.map(demographic -> demographic.getUserValue()).get();
	    LOGGER.debug("PatientID found: " + patientID);
	    
	    LOGGER.info("Storing the PatientID in the consent data structure.");
		consent.setPatientID(patientID);
		acknowledgeRepository.save(consent);
		//delete patient data after storing the Acknowledgment with patientID
        patientRepository.deleteById(notice.getEncounterID());
	}
}
