package eu.europa.ec.healtheid.controllers;


import eu.europa.ec.healtheid.adaptor.model.HealtheidCountryConfiguration;
import eu.europa.ec.healtheid.exception.HealtheidGetCountryConfigurationException;
import eu.europa.ec.healtheid.exception.PatientNotFoundException;
import eu.europa.ec.healtheid.management.WorkflowManager;
import eu.europa.ec.healtheid.mapper.HealtheidCountryConfigurationMapper;
import eu.europa.ec.healtheid.models.Encounter;
import eu.europa.ec.healtheid.models.NotificationData;
import eu.europa.ec.healtheid.models.SearchFieldsForm;
import eu.europa.ec.healtheid.models.enumerator.NotificationType;
import eu.europa.ec.healtheid.utils.FileUtils;
import eu.europa.ec.healtheid.utils.SearchFieldsFormUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.bind.support.SessionStatus;
import org.springframework.web.servlet.ModelAndView;

import java.io.IOException;

import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.*;

/**
* This Controller handles the interactions between HeiD and Patient
* 
* All the Exceptions are handled by HealtheidExceptionHandler.class in eu.europa.ec.healtheid.config
* 
* @author Giuseppe Burrai - Politecnico di Torino
*
*/
@Controller
@RequestMapping(PATIENT_BASE_URL)
@SessionAttributes("token")
public class PatientController {

    private static final Logger LOGGER = LoggerFactory.getLogger(PatientController.class);

    @Autowired
    WorkflowManager workflowManager;

    @Autowired
    FileUtils fileUtils;

    @Value("${acknowledge.path}")
    private String acknowledgePath;

    @Value("${pin.reference.version}")
    private String PINReference;

    //@Value("#{systemProperties['some.key'] ?: 'default system property value'}")
    @Value("${redirect.to.eidasHProxy}")
    private String eidasHProxyURL;

    /**
	 * Method responsible for accepting the Encounter. This endpoint receives the encounterID (JWT Token)
	 * from Patient side. In fact, when patient clicks on the link received by email, it will be
	 * redirected to this endpoint.
	 * 
	 * @param token - the encounterID generated by WorkflowManager to represent the encounter
	 * 
	 * @return redirect the patient to eIDAS HProxy
	 * 			
	 */
    @GetMapping(PATIENT_ACCEPT_ENCOUNTER_URL)
    public ModelAndView acceptEncounter(ModelMap model, @PathVariable(value = "token") String token) {

        boolean valid = workflowManager.validateToken(token);
        if (!valid) return new ModelAndView("errorPage");

        return workflowManager.eIDASredirection(model, token);
    }


    /**
	 * If the eIDAS infrastructure cannot provide the PatientID or other data needed by OpenNCP,
	 * the patient must add the missing data by hand. 
	 * This method retrieve the HealtheidCountryConfiguration, that specifies the OpenNCP data to
	 * fulfill and then redirect the patient towards AdditionalData Screen to add this data.
	 * 
	 * @param token - the encounterID generated by WorkflowManager to represent the encounter
	 * 
	 * @return redirect the patient to additionalData Screen
	 * 			
	 */
    @GetMapping(PATIENT_DATA_URL)
    public ModelAndView additionalPatientData( ModelMap model, @ModelAttribute(value="token") String token) 
    		throws HealtheidGetCountryConfigurationException {

    	LOGGER.info("token: " + token );
        LOGGER.info("PatientID: not found.");

        HealtheidCountryConfiguration healtheidCountryConfiguration = workflowManager.getCountryConfiguration(token);

        SearchFieldsForm searchFieldsForm = HealtheidCountryConfigurationMapper.searchFieldsToSearchFieldsForm(healtheidCountryConfiguration);
        SearchFieldsFormUtils.prepareForView(searchFieldsForm);

        model.addAttribute("searchFieldsForm", searchFieldsForm);
        model.addAttribute("searchFields", healtheidCountryConfiguration.getPatientSearch().getCountry().getSearchFields());

        return new ModelAndView("additionalData", model);
    }

    /**
	 * This method accepts the Patient Additional Data
	 * 
	 * @param token - the encounterID generated by WorkflowManager to represent the encounter
	 * @param searchFieldsForm - patient attributes added by hand 
	 * 
	 * @return redirect the patient to additionalData Screen if some data are missing
	 * 		   otherwise it redirect the patient to Consent Screen
	 * 			
	 */
    @PostMapping(PATIENT_DATA_POST_URL)
    public ModelAndView postAdditionalPatientData(@ModelAttribute(value = "token") String token,
                                                  @ModelAttribute(value = "searchFieldsForm") SearchFieldsForm searchFieldsForm,
                                                  ModelMap model) throws PatientNotFoundException, HealtheidGetCountryConfigurationException {

        if (searchFieldsForm.isValid()) {
            SearchFieldsFormUtils.processViewReturn(searchFieldsForm);
            workflowManager.putAdditionalPatientData(token, searchFieldsForm);
            return new ModelAndView("redirect:"+PATIENT_BASE_URL+PATIENT_ACKNOWLEDGE_URL, model);
        } else {
            HealtheidCountryConfiguration healtheidCountryConfiguration = workflowManager.getCountryConfiguration(token);
            model.addAttribute("searchFieldsForm", searchFieldsForm);
            model.addAttribute("searchFields", healtheidCountryConfiguration.getPatientSearch().getCountry().getSearchFields());

            return new ModelAndView("additionalData", model);
        }
    }
    
    /**
	 * This method redirect the Patient to acknownledge Screen
	 * 
	 * @param token - the encounterID generated by WorkflowManager to represent the encounter
	 * @param encounter
	 * 
	 * @return redirect to PATIENT_ACKNOWLEDGE_POST_URL to store the acknowledge in DB
	 * 			
	 */
    @RequestMapping(value = PATIENT_ACKNOWLEDGE_URL, method = {RequestMethod.GET, RequestMethod.POST})
    public String patientAcknowledge(ModelMap model,
            @ModelAttribute(value="token") String token,
                                 @ModelAttribute(value = "encounter") Encounter encounter
    ) throws IOException {

        if (encounter == null || encounter.isConsent()) return "acknowledgeResult";
        /*		workflowManager.getEncounter(additionalDataForm.getToken());*/

        model.addAttribute("acknowledgeContent", fileUtils.loadFileAsString(acknowledgePath).replaceAll("PIN_REF_VERSION",PINReference));
        model.addAttribute("encounter", encounter);

        return "acknowledge";
    }

    /**
	 * This method handles the request to store Patient Acknowledge in the DB
	 * 
	 * @param encounter
	 * 
	 * @return redirect the Patient to acknowledge Result Screen
	 * 			
	 */
    @PostMapping(PATIENT_ACKNOWLEDGE_POST_URL)
    public String acknowledge(@ModelAttribute(value = "encounter") Encounter encounter,
                          SessionStatus status) throws Exception {

        LOGGER.info("Patient Acknowledge: " + encounter.isConsent());
        LOGGER.info("token : " + encounter.getToken());

        workflowManager.acknowledgeStore(encounter.getToken(), PINReference);
        
        //send notice to Patient about consent status
        NotificationData notice= new NotificationData();
        notice.setEncounterID(encounter.getToken());
        notice.setNotificationType(NotificationType.CONSENT);
        workflowManager.notifyPatient(notice);     

        status.setComplete();

        return "acknowledgeResult";
    }
}
