package eu.europa.ec.healtheid.controllers;

import eu.europa.ec.healtheid.exception.PatientDataNotReadyException;
import eu.europa.ec.healtheid.exception.PatientNotFoundException;
import eu.europa.ec.healtheid.management.WorkflowManager;
import eu.europa.ec.healtheid.models.Encounter;
import eu.europa.ec.healtheid.models.NotificationData;
import eu.europa.ec.healtheid.models.PatientData;
import eu.europa.ec.healtheid.models.enumerator.NotificationType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.*;

import static eu.europa.ec.healtheid.config.HealtheidConnectorConstants.*;

import javax.annotation.Nullable;


/**
* Controller to manage HeiD Client requests
* 
* All the Exceptions are handled by HealtheidExceptionHandler.class in eu.europa.ec.healtheid.config
* 
* @author Giuseppe Burrai - Politecnico di Torino
*
*/

@RestController
@RequestMapping(HEID_CONNECTOR_BASE_URL)
public class HeidClientController {

	
	private static final Logger LOGGER = LoggerFactory.getLogger(HeidClientController.class); 
	@Value("${redirect.to.eidasHProxy}")
	private String eidasHProxyURL;
	
	@Autowired
	WorkflowManager workflowManager;


	/**
	 * Method responsible to receive the ecnounter request from HeiD Client
	 * 
	 * @param encounber - data object that contains the patient phone/email
	 * 
	 * @return encounter - Status 200 OK
	 * 		   the same object with the field encounterID that uniquely represents the encounter
	 * 			
	 */
	@RequestMapping(value = HEID_CONNECTOR_CREATE_ENCOUNTER_URL, method = RequestMethod.POST,
			consumes = MediaType.APPLICATION_JSON_VALUE,
			produces = MediaType.APPLICATION_JSON_VALUE)
	@ResponseStatus(value = HttpStatus.CREATED)
	public ResponseEntity<Encounter> createEncounter(@RequestBody Encounter encounter) throws Exception {
				
		encounter = workflowManager.createEncounter(encounter);

		return new ResponseEntity<>(encounter, HttpStatus.CREATED) ;
	}

/*	@RequestMapping(value = "/testSecurity", method = RequestMethod.POST,
			consumes = MediaType.APPLICATION_JSON_VALUE,
			produces = MediaType.APPLICATION_JSON_VALUE)
	@ResponseStatus(value = HttpStatus.OK)
	public ResponseEntity<Encounter> testSecurity(@RequestBody Encounter encounter) throws Exception {
		encounter = workflowManager.createEncounter(encounter);

		return new ResponseEntity<>( encounter, HttpStatus.CREATED) ;
	}*/
	
	/**
	 * This method handles the PatientData requests. HeiD Client polls this endpoint to check if
	 * some patient data are available.
	 * 
	 * The encounterID is the token that uniquely identifies the encounter. 
	 * It is sent in the HTTP Authorization Header
	 * 
	 * @return patient - the patient data
	 * 
	 * @throws PatientDataNotReadyException if the encounterID exist but Patient Data are not available
	 * @throws PatientNotFoundException if the encounterID not exist 
	 * @throws Exception - if some Internal Error occurs
	 * 		   
	 * 			
	 */
	@RequestMapping(value = HEID_CONNECTOR_PATIENT_DATA_URL, method = RequestMethod.POST,
			consumes = MediaType.APPLICATION_JSON_VALUE,
			produces = MediaType.APPLICATION_JSON_VALUE)
	@ResponseStatus(value = HttpStatus.OK)
	public ResponseEntity<PatientData> requestPatientAuthNData(@RequestHeader("Authorization") String encounterID) 
			throws PatientDataNotReadyException, PatientNotFoundException,Exception {

		PatientData patient = workflowManager.checkIfPatientIsAuthenticated(encounterID);

		return new ResponseEntity<>( patient, HttpStatus.OK) ;
	}
	
	/**
	 * This method manages the notification sent by the HeiD Client that the WM has to forward
	 * to the Patient.
	 * 
	 * The encounterID is the token that uniquely identifies the encounter. 
	 * It is sent in the HTTP Authorization Header
	 * 
	 * @param notice in the HTTP Body. It contains the kind of notification (XCPD,EP,PS,eD)
	 * 
	 * @return Status 200 OK
	 * 
	 * @throws Exception - if something happened during the notification process
	 * 		   
	 * 			
	 */
	@PostMapping(HEID_CONNECTOR_PATIENT_NOTIFY_URL)
	@ResponseStatus(value = HttpStatus.OK)
	public ResponseEntity<Object> receiveNotice(@RequestHeader("Authorization") String encounterID,
			@RequestBody NotificationData notice, ModelMap model) throws Exception {
		
		LOGGER.info(notice.getNotificationType().name().toString()+" successful retrived from openNCP ");
		
		notice.setEncounterID(encounterID);
		
		if (notice.getNotificationType()== NotificationType.XCPD)
			//end of XCPD, update consent with PatientID confirmed by OpenNCP
			workflowManager.updateAcknlowledgement(notice);
	
		workflowManager.notifyPatient(notice);
		
		return new ResponseEntity<>(HttpStatus.OK) ;
	}
}
