package eu.eidas.sp;


import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import com.google.common.collect.ImmutableSortedSet;

import eu.eidas.auth.commons.EidasStringUtil;
import eu.eidas.auth.commons.attribute.AttributeDefinition;
import eu.eidas.auth.commons.attribute.ImmutableAttributeMap;
import eu.eidas.auth.commons.protocol.IRequestMessage;
import eu.eidas.auth.commons.protocol.eidas.LevelOfAssurance;
import eu.eidas.auth.commons.protocol.eidas.LevelOfAssuranceComparison;
import eu.eidas.auth.commons.protocol.eidas.SpType;
import eu.eidas.auth.commons.protocol.eidas.impl.EidasAuthenticationRequest;
import eu.eidas.auth.commons.protocol.impl.EidasSamlBinding;
import eu.eidas.auth.commons.protocol.impl.SamlBindingUri;
import eu.eidas.auth.engine.ProtocolEngineFactory;
import eu.eidas.auth.engine.ProtocolEngineI;
import eu.eidas.auth.engine.SamlEngineSystemClock;
import eu.eidas.auth.engine.configuration.dom.ProtocolEngineConfigurationFactory;
import eu.eidas.auth.engine.metadata.EidasMetadataParametersI;
import eu.eidas.auth.engine.metadata.EidasMetadataRoleParametersI;
import eu.eidas.auth.engine.metadata.MetadataFetcherI;
import eu.eidas.auth.engine.metadata.MetadataSignerI;
import eu.eidas.auth.engine.metadata.MetadataUtil;
import eu.eidas.auth.engine.xml.opensaml.SAMLEngineUtils;
import eu.eidas.engine.exceptions.EIDASMetadataException;
import eu.eidas.engine.exceptions.EIDASSAMLEngineException;
import eu.europa.ec.healtheid.eidashproxy.services.EidasMappingService;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;

@Controller
@RequestMapping("/eidas-hproxy")
public class AuthRequest {

	private static final Logger LOGGER = LoggerFactory.getLogger(AuthRequest.class);

    @Value("${eidas.path}")
    private String eidasConfigPath;
    
    @Value("${sp.metadata.url}")
    private String metadataUrl;
    
    @Value("${provider.name}")
    private String providerName;
    
    @Value("${sp.country}")
    private String spCountry;
    
    @Value("${country.metadata.url}")
    private String countryMetadataUrl;
    
	private ProtocolEngineConfigurationFactory SpProtocolEngineConfigurationFactory;
    
	private ProtocolEngineFactory SpProtocolEngineFactory;

	private ProtocolEngineI protocolEngine;

	private String samlRequest;

	private String nodeMetadataUrl;

	private String nodeUrl;
	
	@Value("${jwt.secret}")
	private String jwtSecret;
		
	@Autowired
	private MetadataFetcherI fetcher;
	
	
	@Autowired
	private EidasMappingService mappingSamlService;

	private void error(String title, String message) {
		throw new ApplicationSpecificServiceException(title, message);
	}

	private String getSSOSLocation(String metadataUrl, SamlBindingUri binding) throws EIDASSAMLEngineException {
		SpProtocolEngineConfigurationFactory = new ProtocolEngineConfigurationFactory(Constants.SP_SAMLENGINE_FILE, null, eidasConfigPath);
		SpProtocolEngineFactory = new ProtocolEngineFactory(SpProtocolEngineConfigurationFactory);
		protocolEngine = SpProtocolEngineFactory.getProtocolEngine("SP");
		MetadataSignerI metadataSigner = (MetadataSignerI) protocolEngine.getSigner();
		EidasMetadataParametersI EidasMetadataParameters;
		try {
			EidasMetadataParameters = fetcher.getEidasMetadata(metadataUrl, metadataSigner, new SamlEngineSystemClock());
			EidasMetadataRoleParametersI EidasMetadataRoleParameters = MetadataUtil.getIDPRoleDescriptor(EidasMetadataParameters);
			Map<String,String> ProtocolBindings = EidasMetadataRoleParameters.getProtocolBindingLocations();

			return ProtocolBindings.get("POST");

		} catch (EIDASMetadataException e) {
			LOGGER.info("Unable to get Eidas Metadata for: " + metadataUrl);
			e.printStackTrace();
			return null;
		}
	}

	private ImmutableAttributeMap RequestedAttributes() {
		final ImmutableSortedSet<AttributeDefinition<?>> allSupportedAttributesSet = protocolEngine.getProtocolProcessor().getAllSupportedAttributes();
		List<AttributeDefinition<?>> reqAttrList = new ArrayList<AttributeDefinition<?>>(allSupportedAttributesSet);
		List<String> reqNameUri = Arrays.asList("http://eidas.europa.eu/attributes/naturalperson/CurrentFamilyName" , "http://eidas.europa.eu/attributes/naturalperson/CurrentGivenName", "http://eidas.europa.eu/attributes/naturalperson/DateOfBirth", "http://eidas.europa.eu/attributes/naturalperson/PersonIdentifier");

		for (AttributeDefinition<?> attributeDefinition : allSupportedAttributesSet) {
			if (!reqNameUri.contains(attributeDefinition.getNameUri().toString()))
				reqAttrList.remove(attributeDefinition);
		}
		return new ImmutableAttributeMap.Builder().putAll(reqAttrList).build();
	}

	@GetMapping("/authentication")
	public ModelAndView startauth(@RequestParam(value="token")String jwtToken) {
		LOGGER.info("start auth request in eidasHProxy");
		byte[] secret256hex; 
		Claims claims;
		String citizen;
		
		secret256hex= DigestUtils.sha256(jwtSecret);
		
		LOGGER.info("Try to read the jwt token.");
		LOGGER.debug("token : " + jwtToken);
		try {
			claims = Jwts.parser()         
				       .setSigningKey(secret256hex)
				       .parseClaimsJws(jwtToken).getBody();		
		} catch(Exception e){
			LOGGER.info("Invalid JWT token. An exception occurs parsing the JWT or checking the signature: ", e.getMessage() );
			throw new ApplicationSpecificServiceException("Could not read jwtToken. "
					+ "An exception occurs parsing the JWT or checking the signature ", e.getMessage());
		}
		
		citizen=claims.getSubject(); //set Patient Country from jwt value
		
		LOGGER.debug("Patient country correctly retrieved from jwt token. Country value: "+ citizen);
		
		LOGGER.info("Retrieve country metadata" + spCountry);
		if (countryMetadataUrl.equals("")) {
			error("Country metadata not found", spCountry + " not in configuration");
		}

		try {
			nodeUrl = getSSOSLocation(countryMetadataUrl, SamlBindingUri.SAML2_POST);
		} catch (EIDASSAMLEngineException e) {
			LOGGER.error("Unable to load SSOSLocation from metadata");
			e.printStackTrace();
		}
		
		LOGGER.info("building eIDAS authentication request");
		EidasAuthenticationRequest.Builder reqBuilder = new EidasAuthenticationRequest.Builder();
		reqBuilder.destination(countryMetadataUrl);
		reqBuilder.providerName(providerName);
		reqBuilder.requestedAttributes(RequestedAttributes());
		reqBuilder.levelOfAssurance(LevelOfAssurance.LOW.stringValue());
		reqBuilder.spType(SpType.PUBLIC.toString());
		reqBuilder.levelOfAssuranceComparison(LevelOfAssuranceComparison.fromString("minimum").stringValue());
		reqBuilder.nameIdFormat("urn:oasis:names:tc:SAML:1.1:nameid-format:unspecified");
		reqBuilder.binding(EidasSamlBinding.EMPTY.getName());
		

		//String metadataUrl = configs.getProperty(Constants.SP_METADATA_URL);
		if (metadataUrl != null && !metadataUrl.isEmpty()) {
			reqBuilder.issuer(metadataUrl);
		}

		reqBuilder.serviceProviderCountryCode(spCountry);
		reqBuilder.citizenCountryCode(citizen);

		IRequestMessage binaryRequestMessage;

		try {
			// TODO quick fix for having a flow working end-to-end check if this is correct:
			// generated missing id
			final String samlid=SAMLEngineUtils.generateNCName();
			reqBuilder.id(samlid);
			binaryRequestMessage = protocolEngine.generateRequestMessage(reqBuilder.build(), nodeMetadataUrl);
		
		} catch (EIDASSAMLEngineException e) {
			LOGGER.error(e.getMessage());
			LOGGER.error("", e);
			throw new ApplicationSpecificServiceException("Could not generate token for Saml Request", e.getMessage());
		}
		
		LOGGER.debug("SAML-id from builder request: " + binaryRequestMessage.getRequest().getId());
		String samlID=binaryRequestMessage.getRequest().getId();
		LOGGER.info("Associating the SAML-ID with the patient token");
		mappingSamlService.storeEncounterID(samlID, jwtToken);
		
		byte[] samltoken = binaryRequestMessage.getMessageBytes();
		samlRequest = EidasStringUtil.encodeToBase64(samltoken);
		nextNode nextNode = new nextNode();
		nextNode.redirectUri = nodeUrl;
		nextNode.postLocationUrl = nodeUrl;
		nextNode.redirectLocationUrl = nodeUrl;
		nextNode.country = citizen;
		nextNode.RelayState = "MyRelayState";
		nextNode.SAMLRequest = samlRequest;
		nextNode.SAMLPlain = EidasStringUtil.toString(samltoken);
		
		ModelAndView model = new ModelAndView("nextnode"); 
		model.addObject("debug", true);
		model.addObject("jwtToken", jwtToken);
		model.addObject(nextNode);
		
		LOGGER.info("Sending request to eIDAS.");
		return model;

	}
}
